#ifndef __UPDATER_H
#define __UPDATER_H

#include "mbed.h"

/** Periodically reads sensor data
 * This class reads and updates sensor data. Intended to be called at a fixed
 * interval.
 * @code
 * Updater *u = Updater::instance(); // beware of lifetime of this pointer
 * Thread updaterThread(osPriorityRealtime, 512, 0, "updater");
 * EventQueue *updaterQueue = mbed_highprio_event_queue();
 * Event<void()> event(updaterQueue, callback(u, &Updater::update));
 * event.period(20);
 * event.post(); // if lifetime of u not correct, this will hard fault
 * updaterThread.start(callback(updaterQueue, &EventQueue::dispatch_forever));  
 *
 */
class Updater: private mbed::NonCopyable<Updater> {
public:
    /// Return singleton instance
    static Updater *instance();
    
    /// Attach a callback handler run each time updater() is run
    void attach(Callback<void()> cb);
    
    /// Update all sensors
    void update(); 
    
    /** Get imu values
     * @param g x, y, z gyro values to return
     * @param a x, y, z accelerometer values to return
     * @param m x, y, z magnetometer values to return
     * @param dt time since data last updated
     * @return g, a, m and dt
     */
    void imu(int g[3], int a[3], int m[3], float& dt);
    
    /** Get gyro values
     * @param g x, y, z gyro values to return
     * @param dt time since data last updated
     * @return g and dt
     */
    void gyro(int g[3], float& dt);

    /** Get accel values
     * @param a x, y, z accelerometer values to return
     * @param dt time since data last updated
     * @return a and dt
     */
    void accel(int g[3], float& dt);
    
    /** Get magnetometer values
     * @param m x, y, z magnetometer values to return
     * @param dt time since data last updated
     * @return m and dt
     */
    void mag(int g[3], float& dt);

    /** Get encoder count
     * @return encoder count since last call
     */
    int encoder();

private:
    /// Basic constructor (singleton)
    Updater();
    
    Callback<void()> _callback; // notification callback    
    Timer *t; // timer used to measure dt
    int _gyro[3]; // gyro raw
    int _accel[3]; // accelerometer raw
    int _mag[3]; // magnetometer raw
    int _ecount; // encoder count
    float _dt;
    int _interval;
    int thisTime;
    int lastTime;
};

#endif