/* NMEA - a small NMEA-parsing library based on TinyGPS

  TinyGPS - a small GPS library for Arduino providing basic NMEA parsing
  Copyright (C) 2008-9 Mikal Hart
  All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  Ported to mbed by Michael Shimniok
*/

#ifdef __MBED__
#include "mbed.h"
#endif

#include <stdint.h>
#include <ctype.h>
#include "GPS.h"

#ifndef __NMEA_h
#define __NMEA_h

#define _GPS_VERSION 9 // software version of this library
#define _GPS_MPH_PER_KNOT 1.15077945
#define _GPS_MPS_PER_KNOT 0.51444444
#define _GPS_KMPH_PER_KNOT 1.852
#define _GPS_MILES_PER_METER 0.00062137112
#define _GPS_KM_PER_METER 0.001

/** An library parsing for parsing select NMEA-0183 sentences
 * @author Michael Shimniok
 */
class NMEA: public GPS {
public:
    typedef uint8_t byte;

    /** Create a new GPS parsing object for parsing NMEA sentences
     */
    NMEA();

    /** Parse a single character received from GPS
     *
     * @param c is the character received from the GPS
     * @returns 1 when all sentences received
     */
    virtual int parse(char c);

#ifndef _GPS_NO_STATS
    void stats(unsigned long *chars, unsigned short *good_sentences, unsigned short *failed_cs);
#endif

    /** determine if all sentences parsed
     *
     */
    inline bool ready() {
        if (_rmc_ready && _gga_ready) {
            _rmc_ready = _gga_ready = false;
            return true;
        } else {
            return false;
        }
    }

    /** Reset the ready flags for all the parsed sentences
     */
    inline void reset_ready() { _rmc_ready = _gga_ready = false; }

    enum {GPS_INVALID_AGE = 0xFFFFFFFF, GPS_INVALID_ANGLE = 999999999, GPS_INVALID_ALTITUDE = 999999999, GPS_INVALID_DATE = 0,
      GPS_INVALID_TIME = 0xFFFFFFFF, GPS_INVALID_SPEED = 999999999, GPS_INVALID_FIX_TIME = 0xFFFFFFFF};

private:
    enum {_GPS_SENTENCE_GPGGA=0x10, _GPS_SENTENCE_GPRMC=0x20, _GPS_SENTENCE_GPGSV=0x40, _GPS_SENTENCE_OTHER=0};

    // properties
    int _new_time;
    int _new_hour;
    int _new_minute;
    int _new_second;
    int _new_date;
    int _new_month;
    int _new_year;
    double _new_latitude;
    double _new_longitude;
    long _altitude;
    float _new_altitude;
    float _new_speed;
    float _new_course;
    float _new_hdop;
    unsigned int _new_sat_count;
    unsigned long _last_time_fix, _new_time_fix;
    unsigned long _last_position_fix, _new_position_fix;

    // parsing state variables
    byte _parity;
    bool _is_checksum_term;
    char _term[15];
    byte _sentence_type;
    byte _term_number;
    uint8_t  _term_offset;
    bool _gps_data_good;
    bool _rmc_ready;
    bool _gga_ready;

    double parse_degrees();
    int from_hex(char a);
    int parse_int();
    double parse_decimal();
    bool term_complete();
    bool gpsisdigit(char c) { return c >= '0' && c <= '9'; }
    int gpsstrcmp(const char *str1, const char *str2);
};

#endif
