/* mbed library for the National Semiconductor ADC128S family of analog to digital converters
 *
 * by Michael Shimniok - http://www.bot-thoughts.com/
 */
 
 /** An interface for driving a National Semiconductor ADC128Sxxx analog to digital converter using SPI
 *
 * @code
 * #include "mbed.h"
 * #include "ADC128S.h"
 *
 * Serial pc(USBTX, USBRX); // tx, rx
 * ADC128S adc(p5, p6, p7, p15); // mosi, miso, sck, cs
 *
 * int main() {
 *   unsigned int result;
 *   pc.baud(115200);    
 *
 *   while(1) {
 *     for (int i = 0; i < 8; i++) {
 *       result = adc.read(i);
 *       pc.printf("ADC(%d)=%d\n", i, result);
 *     }
 *     pc.printf("\n");
 *   }
 * }
 * @endcode
 */
class ADC128S {
public:
    /** Create an ADC128S interface
     *
     * @param mosi  MOSI line
     * @param miso  MISO line
     * @param sck   SCK line
     * @param cs    !CS/!SS line
     */
    ADC128S(PinName mosi, PinName miso, PinName sck, PinName cs);


    /** Get the next channel to be converted with next call to read()
     *
     * @returns an integer representing the adc channel about to be converted
     */ 
    int getChannel(void);

    /** Set channel to be used for the next conversion.
     *
     * @param channel is the adc channel you want to read with next call to read()
     */
    void setChannel(int channel);

    /** Convert the current channel and return the result and prepare to read the next channel.
     *  The channel will count to 7 and then wrap around to 0.  See also: setChannel()
     *
     * @param returns conversion for the curent channel and prepares to read the next channel
     */
    unsigned int read(void);

    /** Read in analog value from the specified ADC channel
     *
     * @param channel  The ADC channel to read
     * @returns the analog to digital conversion result
     */
    unsigned int read(int channel);

private:
    SPI _adc;
    DigitalOut _cs;
    int _channel;
};