/**
 * @file ESP8266.h
 */
#ifndef __ESP8266_H__
#define __ESP8266_H__

#include "mbed.h"
#include "time.h"
#include "ArduinoSerial.h"

/**
 * The way of SoftAP encrypstion. 
 * 
 */
typedef enum {
    ESP8266_ECN_OPEN          = 0, /**< OPEN */
    ESP8266_ECN_WEP           = 1, /**< WEP */
    ESP8266_ECN_WAP_PSK       = 2, /**< WAP_PSK */
    ESP8266_ECN_WAP2_PSK      = 3, /**< WAP2_PSK */
    ESP8266_ECN_WAP_WAP2_PSK  = 4, /**< WAP_WAP2_PSK */
} ESP8266Encrypstion;

/**
 * The ip protocol for communication. 
 * 
 * Only TCP or UPD supported. 
 */
typedef enum {
    ESP8266_COMM_UDP              = 0, /**< UPD */
    ESP8266_COMM_TCP              = 1, /**< TCP */
} ESP8266CommType; 

/**
 * The work mode. 
 * 
 * Station | SoftAP | Station + SoftAP. 
 */
typedef enum {
    ESP8266_MODE_STA         = 1, /**< Station */
    ESP8266_MODE_SAP         = 2, /**< SoftAP */
    ESP8266_MODE_STA_SAP     = 3, /**< Station + SoftAP */
} ESP8266WorkMode;

/**
 * Mux single or multiple connection mode. 
 */
typedef enum {
    ESP8266_MUX_SINGLE      = 0,   /**< Single mode */
    ESP8266_MUX_MULTIPLE    = 1,   /**< Multiple mode */
} ESP8266Mux;

/**
 * Provide methods to manipulate ESP8266 for internet access via WiFi. 
 */
class ESP8266
{
public:

    /**
     * Constructor. 
     * 
     * Need an serial port for communication between MCU and ESP8266. 
     * 
     * @param tx - the TX of serial port. 
     * @param rx - the RX of serial port. 
     * @param baud_rate - the baud rate of communication(default 9600).
     */
    ESP8266(PinName tx, PinName rx, int baud_rate = 9600);
    
    /**
     * Set as station mode and join in ssid with pwd. 
     *
     * @param ssid - SSID of AP to join in. 
     * @param pwd - PASSWORD of AP to join in. 
     * @return true if success, false for failure. 
     */
    bool setStationMode(String ssid, String pwd);
    
    /**
     * Set as softap mode and start softap. 
     *
     * @param sap_ssid - SSID of its self softap. 
     * @param sap_pwd - PASSWORD of its self softap. 
     * @param chl - the channel can be 1, 2, 3, ..., 13(default: 7).  
     * @param enc - the method of encrypstion(default: ESP8266_ECN_WAP_WAP2_PSK). 
     * @return true if success, false for failure. 
     */
    bool setSoftAPMode(String sap_ssid, String sap_pwd,
        int chl = 7, ESP8266Encrypstion ecn = ESP8266_ECN_WAP_WAP2_PSK);
    
    /**
     * Set as softap + station mode and start softap and join in ssid with pwd. 
     *
     * @param ssid - SSID of AP to join in. 
     * @param pwd - PASSWORD of AP to join in. 
     * @param sap_ssid - SSID of its self softap. 
     * @param sap_pwd - PASSWORD of its self softap. 
     * @param chl - the channel can be 1, 2, 3, ..., 13(default: 7).  
     * @param enc - the method of encrypstion(default: ESP8266_ECN_WAP_WAP2_PSK). 
     * @return true if success, false for failure. 
     */
    bool setStationSoftAPMode(String ssid, String pwd, String sap_ssid, String sap_pwd ,
        int chl = 7, ESP8266Encrypstion ecn = ESP8266_ECN_WAP_WAP2_PSK);
    
    /**
     * Start a TCP connection to server or init UDP communication. 
     * 
     * @param type - TCP or UDP.
     * @param addr - ip address or domain name of server to connect. 
     * @param port - the port number of server to connect. 
     * @param mux - single or mulitple connection mode(default: ESP8266_MUX_SINGLE). 
     * @param id - id number when mux is multiple(default: 0). 
     * @return true for success, false for failure.
     */
    bool ipConfig(ESP8266CommType type, String addr, int port, ESP8266Mux mux = ESP8266_MUX_SINGLE, int id = 0);
    
    /**
     * Set the connection mode(sigle or multiple)
     *
     * @param mux - single or multiple. 
     * @return true for success, false for failure.
     */
    bool confMux(ESP8266Mux mux);    
    
    /**
     * Start or stop the TCP server.
     * 
     * @param mode - 1 to start, 0 to stop (restart needed). 
     * @param port - the port to listen. 
     * @return true for success, false for failure.
     */
    bool confServer(int mode, int port);  
    
    /**
     * Send data in sigle connection mode(mux = 0). 
     * 
     * @param str - string to send. 
     * @return true for success, false for failure.
     */
    bool send(String str);  
    
    /**
     * Send data in multiple connection mode(mux = 1). 
     *
     * @param id - id number. 
     * @param str - string to send by id connection. 
     * @return true for success, false for failure.
     */
    bool send(int id, String str);  
    
    /**
     * Start to receive data from network. 
     * 
     * @param buf - buffer for storing data from network. 
     * @return the size of data received.
     */
    int recvData(char *buf);
    
    /**
     * Get the current id of data received via member method:recvData in multiple connection mode.
     *
     * @return the id of connection which data received from. 
     */
    int getMuxID(void);
    
    /** 
     * Reset the module. 
     *
     * @return true for success, false for failure.
     */
    bool reset(void);    
    
    /**
     * Get current work mode. 
     * 
     * @return the mode. 
     */
    String showMode(void);   
    
    
    /**
     * Get the list of Access Points available. 
     *
     * @return the list of SSID. 
     */
    String showAP(void);   
    
    /**
     * Get the SSID of AP joined already. 
     * 
     * @return the SSID joined. 
     */
    String showJAP(void);  
    
    /**
     * Quit the AP joined before. 
     * 
     * @return true for success, false for failure.
     */
    bool quitAP(void);    
    
    
    /**
     * Get SoftAP Parameters. 
     *
     * @return the parameters of SoftAP.
     */
    String showSAP(void);     

    /**
     * Get the status of connection. 
     *
     * @return the status. 
     */
    String showStatus(void);  
    
    /**
     * Get the connection mode(sigle or multiple)
     * 
     * @return the mux string. 
     */
    String showMux(void);  
    
    /**
     * Show the current ip address. 
     * 
     * @return the ip address. 
     */
    String showIP(void);    
    
    /**
     * Create new tcp or udp connection in sigle connection mode. 
     *
     * @param type - UDP or TCP.
     * @param addr - the ip or domain name of server. 
     * @param port - the port number. 
     */
    bool newMux(ESP8266CommType type, String addr, int port);   
    
    /**
     * Create new tcp or udp connection in multiple connection mode)(id:0-4). 
     *
     * @param id - the id number. 
     * @param type - UDP or TCP.
     * @param addr - the ip or domain name of server. 
     * @param port - the port number.      
     */
    bool newMux(int id, ESP8266CommType type, String addr, int port);   
    
    /**
     * Close tcp or udp in sigle connection mode.  
     */
    void closeMux(void);   
    
    /**
     * Close tcp or udp in multiple connection mode. 
     *
     * @param id - the id number.
     */
    void closeMux(int id); 

private:

    /**
     * Set work mode. 
     * 
     */
    bool confMode(ESP8266WorkMode mode);   
    
    
    /**
     * Join in AP. 
     * 
     */
    bool confJAP(String ssid , String pwd);    
    
    
    /**
     * Set SoftAP Parameters. 
     *
     */
    bool confSAP(String ssid , String pwd , int chl , int ecn);      
        
     
    ArduinoSerial m_uart; /* The UART to communicate with ESP8266 */
    int m_mux_id;
};

#endif /* #ifndef __ESP8266_H__ */