/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include "platform/mbed_thread.h"
#include "stdio.h"


#define HEIGHT 20
#define WIDTH  60

//char trailbuffer[WIDTH][HEIGHT];
/* prototype of function */
struct trail {
    int start;
    int end;
    int speed;
    int pos;
    char matStr[HEIGHT];
    int show;
} trails[WIDTH];
void initTrail( struct trail *trail ) ;
// select graphic line drawing mode default to UTF-8
//#define VT100
//#define POOR_MANS
#define UTF8 // select vt100 line drawing set
#include "VT100.h"

// Page elements positions
#define TLINE   4  // line where temperature values are displayed
#define LLINE   6  //   "    "   light......          
#define TLPOS   2  // Temperature  and light level displays start on this line
#define FCOL    3  // First column for text
#define TLVAL  25  // Start sensor reading displays in this column
#define TLLED  32  // column for starting led simulations display
#define TLLTH  39  // column for start of display of low threshold values 
#define TLHTH  56  // column for start of display of high threshold values 
#define HEATL   8  // Status display line for the heater
#define LIGHTL 10  // Status display line for lighting
#define STATUS 12  // Status line for user input response
#define USRHLP 14  // line where user help is displayed
#define MODEC  18  // column start for lighting and heating mode Auto/Manual
#define ONOFF  27  // column start for on or off status
#define INSTR  35  // column start for toggling instruction
#define RTCCOL 43  // column Start for Real Time Clock

// Blinking rate in milliseconds
#define BLINKING_RATE_MS 500
#define SW2 P0_4
// Specify different pins to test printing on UART other than the console UART.
#define TARGET_TX_PIN USBTX
#define TARGET_RX_PIN USBRX

// page setup
#define PAGEWIDTH 70
#define PAGELENGTH 20

/*******************************************************************************
 *
 * page[][] 2d representation of the screen border display
 *
 *
 * j - Bottom right _|
 * k - Top right corner
 * l - top left corner
 * m - Bottom left |_
 * n - cross +
 * q - horizontal bar
 * t - left hand |-
 * u - right hand -|
 * v - Bottom with line up
 * w - top with line down
 * x - vertical bar |
 *
 ******************************************************************************/
char page[PAGELENGTH][PAGEWIDTH] = {
    "lqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqwqqqqqqqqqqqqqqqwqqqqqqqqqqqqqqqqk\r\n",
    "x                               x               x                x\r\n",
    "tqqqqqqqqqqqqqqqqqqqqqwqqqqqqqwqnqqqqqqqqqqqqqqqnqqqqqqqqqqqqqqqqu\r\n",
    "x                     x       x x               x                x\r\n",
    "tqqqqqqqqqqqqqqqqqqqqqnqqqqqqqnqnqqqqqqqqqqqqqqqnqqqqqqqqqqqqqqqqu\r\n",
    "x                     x       x x               x                x\r\n",
    "tqqqqqqqqqqqqqqwqqqqqqvqwqqqqqnqnqqqqqqqqqqqqqqqvqqqqqqqqqqqqqqqqu\r\n",
    "x              x        x     x x                                x\r\n",
    "tqqqqqqqqqqqqqqnqqqqqqqqnqqqqqnqnqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqu\r\n",
    "x              x        x     x x                                x\r\n",
    "tqqqqqqqqqqqqqqvqqqqqqqqvqqqqqvqvqqqqwqqqqqqqqqqqqqqqqqqqqqqqqqqqu\r\n",
    "x                                    x                           x\r\n",
    "tqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqvqqqqqqqqqqqqqqqqqqqqqqqqqqqu\r\n",
    "x                                                                x\r\n",
    "x                                                                x\r\n",
    "x                                                                x\r\n",
    "x                                                                x\r\n",
    "x                                                                x\r\n",
    "x                                                                x\r\n",
    "mqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqj\r\n",
};
// Create a BufferedSerial object to be used by the system I/O retarget code.
static Serial serial_port(TARGET_TX_PIN, TARGET_RX_PIN, 115200);

// Global variables
char buffer[80];
Thread thread;
bool allowUpdate = true;

time_t currentTime;

FileHandle *mbed::mbed_override_console(int fd)
{
    return &serial_port;
}

//Wifi Feature
#include "ntp-client/NTPClient.h"
WiFiInterface *wifi;

// Initialise the digital  and anaolgue input and output pins
DigitalOut led(LED1);
DigitalIn pushButton(SW2, PullUp);
// Wing connections
AnalogIn vTherm(P10_1);
// breadboard connections
AnalogIn lightLevel(P10_4);
DigitalOut boilerLed(P10_5);
DigitalOut lightingLed(P10_0);
DigitalOut yellowLed(P0_5);

/* Data Structures */

struct dataSet {
    float highTempThresh = 26.0;
    float lowTempThresh = 23.0;
    float ambientTemp;
// Heating and Lighting controls
    bool heatingStatus = 0;  //  off = 0,  on = 1
    bool heatingMode = 0;    // auto = 0, man = 1
    float highLightThresh = 80.0;
    float lowLightThresh = 20.0;
    float ambientLight;
    bool lightingStatus = 0; //  off = 0,  on = 1
    bool lightingMode = 0;   // auto = 0, man = 1
    int timeStatus = 0;
    int wifiStatus = 0;
} myData;

/* prototype of function */
void posCursor(int col, int line);
void displayAt( int col, int line, char *strBuffer );
void initialise();
void readSensors();
void setActuators();
void displayData();
void drawBorders();
void updateRealTimeClock(char *timeBuffer);
void matrix();

/*******************************************************************************
 *
 * console keyboard scanning thread
 *
 * Started after reset
 *
 * Checks for characters from the serial link, decodes the requested action:
 *
 * '<sp>'  Set max and minimum threshold values for heating temperature and
 *         light levels.
 * '<tab>' Select between individual thresholds to adjust.
 * 'MH'    Toggle Auto/Manual for heating
 *     'H' when in manual mode toggle heating on or off
 * 'ML'    Toggle Auto/Manual for lighting
 *     'L' when in manual mode toggle lighting on or off
 * 'T'     Set the current Time and Date via Wi-Fi
 * 'S'     Start Self-Test Function
 * 'R'     Redraw Static Text and Borders
 * 
 * simple line editor to accept values typed on keyboard
 *
 */
void consoleThread()
{
    char inputStr[20];
    char inputChar;
    bool toggling = false;
    bool settingTime = false;
    bool threshSetting = false;
    int fieldSize = 32;
    int index = 0; // character input index in inputStr

    int selection = 0; //{tempLow==0, tempHigh==1, lightLow==2, lightHigh==3}
    int selcoords[4][2];  // Coordinates for the threshold setting fields

    selcoords[0][0] = TLINE;  // line
    selcoords[0][1] = TLLTH;  // column
    selcoords[1][0] = TLINE;
    selcoords[1][1] = TLHTH;
    selcoords[2][0] = LLINE;
    selcoords[2][1] = TLLTH;
    selcoords[3][0] = LLINE;
    selcoords[3][1] = TLHTH;
    while(1) {
        if (serial_port.readable()) {
            allowUpdate = false;
            inputChar = serial_port.getc();
            switch (inputChar) {
                case 0x08:  // backspace
                    toggling = false;

                    if (index > 0) {
                        inputStr[index-1] = ' ';
                    }

                    if (settingTime) {
                        displayAt( RTCCOL, STATUS, inputStr );
                        posCursor( RTCCOL+index-1, STATUS );
                    } else {
                        displayAt(selcoords[selection][1], selcoords[selection][0], inputStr);
                        posCursor( selcoords[selection][1]+index-1, selcoords[selection][0] );
                    }
                    inputStr[--index] = NULL;

                    break;
                case 0x09:  // tab key
                    toggling = false;
                    if (threshSetting) {
                        index = 0;
                        inputStr[index] = NULL;
                        switch (selection) {// currently selected threshold
                            case 0: // change focus from low thresh temperature
                                sprintf(buffer, "%2.1f", myData.lowTempThresh);
                                BLACK_BACK;
                                WHITE_TEXT;
                                displayAt( TLLTH, TLINE, buffer);
                                WHITE_BACK;
                                BLACK_BOLD;  // to high thresh temperature
                                sprintf(buffer, "%2.1f", myData.highTempThresh);
                                displayAt( TLHTH, TLINE, buffer);
                                YELLOW_BOLD;
                                posCursor( TLHTH, TLINE );

                                break;
                            case 1: // change focus from high thresh temperature
                                sprintf(buffer, "%2.1f", myData.highTempThresh);
                                BLACK_BACK;
                                WHITE_TEXT;
                                displayAt( TLHTH, TLINE, buffer);
                                WHITE_BACK;
                                BLACK_BOLD;  // to low thresh light
                                sprintf(buffer, "%2.1f", myData.lowLightThresh);
                                displayAt( TLLTH, LLINE, buffer);
                                YELLOW_BOLD;
                                posCursor( TLLTH, LLINE );

                                break;
                            case 2: // change focus from low thresh light
                                sprintf(buffer, "%2.1f", myData.lowLightThresh);
                                BLACK_BACK;
                                WHITE_TEXT;
                                displayAt( TLLTH, LLINE, buffer);
                                WHITE_BACK;
                                BLACK_BOLD;  // to high thresh light
                                sprintf(buffer, "%2.1f", myData.highLightThresh);
                                displayAt( TLHTH, LLINE, buffer);
                                YELLOW_BOLD;
                                posCursor( TLHTH, LLINE );

                                break;
                            case 3: // change focus from high thresh light
                                sprintf(buffer, "%2.1f", myData.highLightThresh);
                                BLACK_BACK;
                                WHITE_TEXT;
                                displayAt( TLLTH, LLINE, buffer);
                                WHITE_BACK;
                                BLACK_BOLD;  // to low thresh temperature
                                sprintf(buffer, "%2.1f", myData.lowTempThresh);
                                displayAt( TLLTH, TLINE, buffer);
                                YELLOW_BOLD;
                                posCursor( TLLTH, TLINE );

                                break;
                        }
                        selection = (selection + 1)%4; // keep it within bounds
                    } else allowUpdate = true;
                    break;
                case 0x0d:  // enter key, accept input
                    if (settingTime) {
                        updateRealTimeClock(inputStr);
                    } else {
                        if (index > 0) {
                            switch (selection) {
                                case 0:
                                    myData.lowTempThresh = atof(inputStr);
                                    break;
                                case 1:
                                    myData.highTempThresh = atof(inputStr);
                                    break;
                                case 2:
                                    myData.lowLightThresh = atof(inputStr);
                                    break;
                                case 3:
                                    myData.highLightThresh = atof(inputStr);
                                    break;
                            }
                        }
                    }
                    toggling = false;
                    settingTime = false;
                    threshSetting = false; // stop every update action and restart display

                    inputStr[index] = NULL;
                    index = 0;
                    allowUpdate = true;
                    BLACK_BACK;
                    displayAt(FCOL, STATUS, (char *)"                                ");
                    HIDE_CURSOR;
                    break;
                case 0x20: // Space bar data coming...
                    SHOW_CURSOR;
                    if (settingTime) {
                        inputStr[index++] = inputChar;
                        inputStr[index] = NULL;
                        displayAt( RTCCOL, STATUS, inputStr );
                    } else if (!threshSetting) { // highlight temp low threshold field
                        fieldSize = 4;
                        toggling = 0; // reset any previous actions in progress
                        selection = 0; // always start at low thresh temp
                        threshSetting = true;
                        displayAt(FCOL, STATUS, (char *)"Change threshold");
                        index = 0;
                        inputStr[index] = NULL;
                        //tempLow:
                        sprintf(buffer, "%2.1f", myData.lowTempThresh);
                        WHITE_BACK;
                        BLACK_BOLD;
                        displayAt( TLLTH, TLINE, buffer);
                        WHITE_BACK;
                        YELLOW_BOLD;
                        posCursor( TLLTH, TLINE );
                    }
                    break;
                case 'h':
                    if (toggling)  {
                        myData.heatingMode = !myData.heatingMode;
                        toggling = false;
                        displayAt(FCOL, STATUS, (char *)"                         ");
                        if (myData.heatingMode == 1) {
                            displayAt(INSTR, HEATL, (char *)"\'H\' to toggle on/off");
                        } else {
                            displayAt(INSTR, HEATL, (char *)"                    ");
                        }
                    } else if (myData.heatingMode == 1) {
                        myData.heatingStatus = !myData.heatingStatus;
                    }
                    allowUpdate = true;
                    break;
                case 'l':
                    if (toggling)  {
                        myData.lightingMode = !myData.lightingMode;
                        toggling = false;
                        displayAt(FCOL, STATUS, (char *)"                         ");
                        if (myData.lightingMode == true) {
                            displayAt(INSTR, LIGHTL, (char *)"\'L\' to toggle on/off");
                        } else {
                            displayAt(INSTR, LIGHTL, (char *)"                    ");
                        }
                    } else if (myData.lightingMode == true) {
                        myData.lightingStatus = !myData.lightingStatus;
                    }
                    allowUpdate = true;
                    break;
                case 'm':
                    toggling = true;
                    displayAt(FCOL, STATUS, (char *)"Press L(ight) or H(eater)");
                    index = 0;
                    allowUpdate = true;
                    break;
                case 'r':  // redraw the static text and borders
                    drawBorders();
                    index = 0;
                    allowUpdate = true;
                    break;
                case 's':  // self test
                    matrix();
                    drawBorders();
                    index = 0;
                    allowUpdate = true;
                    break;
/*******************************************************************************
 *
 * Set time Via WiFi
 *
 * Load up connection screen
 * Connect to wifi
 * Fetch time and date
 * Set the time and date
 * Disconnect from WiFi
 * Light Time Status LED 
 * Load up HMI
 *
 ******************************************************************************/                
                case 't': { // set time for real time clock
                    printf("\033[2J\033[H"); // clear screen and move the cursor to 1, 1
                    printf("\033[?25l"); // Turn off visible cursor
                    printf( "%c\033[0m", 0x0f ); // Reset all styles
                    printf( "\033[32m" ); // bright green text
                    printf("Please Wait, Connecting to WiFi...");

                    wifi = WiFiInterface::get_default_instance();  // connect to wifi
                    // myDatat.wifiStatus is == 0 if successful
                    myData.wifiStatus = wifi->connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2);
                    NTPClient ntpclient(wifi);                     // connect NTP Client
                    time_t timestamp = ntpclient.get_timestamp();  // temporary store for NTP time
                    if (timestamp > 0) {                // timestamp is valid if not less than 0
                        set_time( timestamp );          // set local time to current network time
                        myData.timeStatus = 1;
                    } else {                            // unable to get ntp time successfully
                        myData.timeStatus = 0;
                    }
                    wifi->disconnect();                   // done
                    yellowLed = myData.timeStatus;        // indicate time status with led
                    //or   yellowLed = !myData.wifiStatus       // indicate wifi connected successfully
                    drawBorders();
                    index = 0;
                    allowUpdate = true;
                    break;
                }
                    default: // should be a digit or decimal point or ; or /
                    toggling = 0;
                    if (index < fieldSize) {  // ignore inputs beyond 4 digits for thresh and 20 for time.
                        inputStr[index++] = inputChar;
                        inputStr[index] = NULL;
                        YELLOW_BOLD;
                        SHOW_CURSOR;
                        if (settingTime) {
                            displayAt(RTCCOL, STATUS, inputStr);
                        } else displayAt( selcoords[selection][1], selcoords[selection][0], inputStr);
                    } else ; //todo handle numbers that are too big for the field: kick an error
                    break;
            }
        }
        thread_sleep_for(5); // let other task do stuff for 5ms
    }
}


int main()
{
    initialise(); // function to setup VT100 display
    thread.start( consoleThread ); // Start the console keyboard input thread
    /****************************************************************************
     *
     * Main loop:-
     *
     *
     * flash status led
     * read sensors to get current environmental conditions
     * set actuators to turn on the appropriate heating/cooling and lighting
     * systems
     * display data read from the sensors and threshold values
     * sleep for 0.5seconds
     *
     ***************************************************************************/

    while (true) {
        led = !led; // flashing status signal
        readSensors();
        setActuators();
        if (allowUpdate) displayData();
        thread_sleep_for(BLINKING_RATE_MS);
    }
}
/*******************************************************************************
 *
 * Read Seonsors Function
 *
 * Read the thermmistor voltage 
 * Calculate the thermistor resistance from the Voltage and Current values
 * Calculate temperatur using Steinhart-Hart Equation
 * Set ambient temp (myData)
 * Set heating status
 * 
 * Calculate ambient light % from light level
 * Set ambient light (myData)
 * Set lighting status
 *
 ******************************************************************************/ 
void readSensors()
{
    /* read thermistor Voltage */
    float refVoltage = vTherm.read() * 2.4; // Range of ADC 0->2*Vref
    float refCurrent = refVoltage  / 10000.0; // 10k Reference Resistor
    float thermVoltage = 3.3 - refVoltage;    // Assume supply voltage is 3.3v
    float thermResistance = thermVoltage / refCurrent;
    float logrT = (float32_t)log((float64_t)thermResistance);

    /* Calculate temperature from the resistance of thermistor using Steinhart-Hart Equation */
    float stEqn = (float32_t)((0.0009032679) + ((0.000248772) * logrT) +
                              ((2.041094E-07) * pow((float64)logrT, (float32)3)));

    myData.ambientTemp = (float32_t)(((1.0 / stEqn) - 273.15)  + 0.5);
    if ((myData.heatingStatus == 0) && (myData.ambientTemp < myData.lowTempThresh) && (myData.heatingMode == 0)) myData.heatingStatus = 1;  // turn on
    if ((myData.heatingStatus == 1) && (myData.ambientTemp > myData.highTempThresh) && (myData.heatingMode == 0)) myData.heatingStatus = 0; // turn off
    myData.ambientLight = ( 1 - lightLevel.read()) * 100;
    if ((myData.lightingStatus == 0) && (myData.ambientLight < myData.lowLightThresh) && (myData.lightingMode == 0)) myData.lightingStatus = 1;  // turn on
    if ((myData.lightingStatus == 1) && (myData.ambientLight > myData.highLightThresh) && (myData.lightingMode == 0)) myData.lightingStatus = 0; // turn off
}

/*******************************************************************************
 *
 * displayData()
 *
 * Update all sensor and actuator mimic fields with current values
 *
 * LED simulation uses reverse video space in the colour of the text field
 *
 ******************************************************************************/
void displayData()
{
//    int tCol, lCol;
    sprintf(buffer, " ");
    REVERSE;
    if (myData.lightingStatus) {
        GREEN_BOLD;
    } else {
        RED_BOLD;
    }
    displayAt(TLLED, LIGHTL, buffer);

    if (myData.ambientTemp > myData.highTempThresh) {
        RED_BOLD;
        displayAt(TLLED, TLINE, buffer); // LED simulation reverse video space
        NORMAL;
        RED_BOLD;
    } else if (myData.ambientTemp < myData.lowTempThresh) {
        BLUE_BOLD;
        displayAt(TLLED, TLINE, buffer);
        NORMAL;
        BLUE_BOLD;
    } else {
        GREEN_BOLD;
        displayAt(TLLED, TLINE, buffer);
        NORMAL;
        GREEN_BOLD;
    }
    sprintf(buffer, "%2.1fC ", myData.ambientTemp);
    displayAt(TLVAL, TLINE, buffer);
    WHITE_TEXT;
    sprintf(buffer, "%2.1fC ", myData.lowTempThresh);
    displayAt(TLLTH, TLINE, buffer);
    sprintf(buffer, "%2.1fC ", myData.highTempThresh);
    displayAt(TLHTH, TLINE, buffer);
    if (myData.heatingMode) {
        CYAN_BOLD;
        sprintf(buffer, "%s", "Manual");
    } else {
        GREEN_BOLD;
        sprintf(buffer, "%s", " Auto ");
    }
    displayAt(MODEC, HEATL, buffer);
    if (myData.heatingStatus) {
        GREEN_BOLD;
        sprintf(buffer, "%s", " ON");
    } else {
        BLUE_BOLD;
        sprintf(buffer, "%s", "OFF");
    }
    displayAt(ONOFF, HEATL, buffer);
    sprintf(buffer, " ");
    REVERSE;
    if (myData.heatingStatus) {
        GREEN_BOLD;
    } else {
        RED_BOLD;
    }
    displayAt(TLLED, HEATL, buffer);

    if (myData.ambientLight > myData.highLightThresh) {
        RED_BOLD;
        displayAt(TLLED, LLINE, buffer);
        NORMAL;
        RED_BOLD;
    } else if (myData.ambientLight < myData.lowLightThresh) {
        BLUE_BOLD;
        displayAt(TLLED, LLINE, buffer);
        NORMAL;
        BLUE_BOLD;
    } else {
        GREEN_BOLD;
        displayAt(TLLED, LLINE, buffer);
        NORMAL;
        GREEN_BOLD;
    }
    sprintf(buffer, "%3.1f%c ", myData.ambientLight, '%' );
    displayAt(TLVAL, LLINE, buffer);
    WHITE_TEXT;
    sprintf(buffer, "%2.1f%c ", myData.lowLightThresh, 0x25);
    displayAt(TLLTH, LLINE, buffer);
    sprintf(buffer, "%2.1f%c ", myData.highLightThresh, 0x25);
    displayAt(TLHTH, LLINE, buffer);
    if (myData.lightingMode) {
        CYAN_BOLD;
        sprintf(buffer, "%s", "Manual");
    } else {
        GREEN_BOLD;
        sprintf(buffer, "%s", " Auto ");
    }
    displayAt(MODEC, LIGHTL, buffer);
    if (myData.lightingStatus) {
        GREEN_BOLD;
        sprintf(buffer, "%s", " ON");
    } else {
        BLUE_BOLD;
        sprintf(buffer, "%s", "OFF");
    }
    displayAt(ONOFF, LIGHTL, buffer);

// Display Time and Date
    char timeBuffer[20]; // buffer for time string

    currentTime = time(NULL);
    strftime(timeBuffer, 32, "%d/%m/%Y %H:%M:%S", localtime(&currentTime));
    WHITE_BOLD;
    displayAt(RTCCOL, STATUS, timeBuffer);
}
/*******************************************************************************
 *
 * setActuators()
 *
 * Control switches/relays for hardware connected to the environmental control
 * system
 *
 ******************************************************************************/
void setActuators()
{
    boilerLed = myData.heatingStatus;
    lightingLed = myData.lightingStatus;
}

/*******************************************************************************
 *
 * displayAt( column, line, string )
 *
 * Move the VT100 cursor to new coordinates on the display and print a string
 * of characters.
 *
 ******************************************************************************/
void displayAt( int col, int line, char *strBuffer )
{
    {
        printf( "\033[%d;%dH%s", line, col, strBuffer);
        fflush(stdout);
    }
}
/*******************************************************************************
 *
 * posCursor( column, line )
 *
 * Move the VT100 cursor to precise coordinates on the display
 *
 ******************************************************************************/
void posCursor(int col, int line)
{

    {
        printf( "\033[%d;%dH", line, col);
        fflush(stdout);
    }
}
/*******************************************************************************
 *
 * initialise()
 *
 * Called once following reset to setup the vt100 screen console display
 *
 ******************************************************************************/

void initialise()
{
    RIS;  // Full terminal reset just in case
    fflush(stdout);
    thread_sleep_for(200);


    drawBorders();  // Borders and static text
}

/*******************************************************************************
 *
 * drawBorders()
 *
 * Converts the grid of lines and corners defined in page[][] to a printable
 * string of characters appropriate to the font of the terminal emulator.
 * UTF8 character set is the most commonly used with vt100 on PUTTY or Teraterm.
 * Each shape is defined by three bytes in UTF8.
 * Other supported modes are POOR_MANS  +, -, | and VT100 one byte graphic
 * characters these are non-standard and not universally supported in all or
 * even many fonts.
 *
 * Prints the Static Text in the appropriate locations on the display
 *
 ******************************************************************************/
void drawBorders()
{
    char outStr[1040];
    char outChar[4];
    int where;
    int charLen;
    CLS;
    HOME; // clear screen and move the cursor to 0, 0
    HIDE_CURSOR; // Turn off visible cursor[?25 lower case L
    WHITE_BOLD;
#ifdef GRAPHIC_SET
    printf("%c", GRAPHIC_SET);
#endif
    for (int line = 0; line < PAGELENGTH; line++) {
        where = 0;
        for (int column = 0; column < PAGEWIDTH; column++) {
            switch (page[line][column]) {
                case 'j':
                    charLen = strlen(HR);
                    strcpy(outChar, HR);
                    for (int i=0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case 'k':
                    charLen = strlen(TR);
                    strcpy(outChar, TR);
                    for (int i=0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case 'l':
                    charLen = strlen(TL);
                    strcpy(outChar, TL);
                    for (int i=0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case 'm':
                    charLen = strlen(HL);
                    strcpy(outChar, HL);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case'n':
                    charLen = strlen(MC);
                    strcpy(outChar, MC);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case'q':
                    charLen = strlen(HZ);
                    strcpy(outChar, HZ);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case't':
                    charLen = strlen(VT);
                    strcpy(outChar, VT);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case'u':
                    charLen = strlen(VR);
                    strcpy(outChar, VR);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case'v':
                    charLen = strlen(HU);
                    strcpy(outChar, HU);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case'w':
                    charLen = strlen(HD);
                    strcpy(outChar, HD);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                case'x':
                    charLen = strlen(VB);
                    strcpy(outChar, VB);
                    for (int i = 0; i < charLen; i++) {
                        outStr[where++] = outChar[i];
                    }
                    break;
                default:
                    outStr[where++] = page[line][column];  // Space or blank is needed
            }
        }
        outStr[where] = NULL;
        printf("%s", outStr);
    }
#ifdef TEXT_SET
    printf("%c", TEXT_SET);
#endif
    WHITE_BOLD; // set text color to white bold
    displayAt( FCOL, TLPOS, (char *) "Environmental Control System");
    BLUE_BOLD;
    displayAt( TLLTH-4, TLPOS, (char *)"Low Threshold");
    RED_BOLD;
    displayAt( TLHTH-5, TLPOS, (char *)"High Threshold");
    CYAN_BOLD;
    displayAt(FCOL, TLINE, (char *)"Temperature");
    displayAt(FCOL, LLINE, (char *)"Ambient Light Level");
    displayAt(FCOL, HEATL, (char *)"Heater");
    displayAt(FCOL, LIGHTL, (char *)"Lighting");
    WHITE_TEXT;
    displayAt(FCOL, USRHLP, (char *)"* Press \"Space key\" to adjust threshold values\r\n");
    displayAt(FCOL, USRHLP+1, (char *)"  - Use \"Tab key\" to select each threshold setting\r\n");
    displayAt(FCOL, USRHLP+2, (char *)"  - Hit \"Enter key\" to store new threshold value");
    displayAt(FCOL, USRHLP+3, (char *)"* Press \"M key\" to change auto/manual modes");
    displayAt(FCOL, USRHLP+5, (char *)"* Press \"T key\" to set time via WiFi");
    displayAt(FCOL, USRHLP+4, (char *)"* Press \"S key\" to start Self Test");
    fflush(stdout); // send the codes to the terminal

}
/*******************************************************************************
 *
 * Update RTC function
 *
 * Take a string from the console thread when the user sets the time.
 * Decodes the string and uses the data to set the time and date.
 *
 * TODO: remove unwanted code due to new WiFi function to get time
 *
 ******************************************************************************/
void updateRealTimeClock(char *timeBuffer)
{
    char *tp;
    char *timeArray[6];
    int arrayIndex;
    struct tm struct_time;

    // extract number from string
    arrayIndex = 0;
    tp = strtok( timeBuffer, " /:-" );
    timeArray[arrayIndex++] = tp;
    while ( tp != NULL && arrayIndex < 6 ) {  // parse the values and assign to time array
        tp = strtok( NULL," /:-" );           // TODO: do some error checking
        timeArray[arrayIndex++] = tp;
    }
    // store number into time struct
    struct_time.tm_mday = atoi(timeArray[0]);
    struct_time.tm_mon  = atoi(timeArray[1]) - 1;
    struct_time.tm_year = atoi(timeArray[2]) - 1900;
    struct_time.tm_hour = atoi(timeArray[3]);
    struct_time.tm_min  = atoi(timeArray[4]);
    struct_time.tm_sec  = atoi(timeArray[5]);

    currentTime = mktime(&struct_time);
    set_time(currentTime);
}
/*******************************************************************************
 *
 * Matrix function (self test)
 *
 * Loads up new screen with green trails of random characters
 * Flashes all status LEDs randomly to show functionality
 *
 ******************************************************************************/
void matrix(void)
{
    srand((unsigned)time(NULL));
    printf("\033[2J\033[H"); // clear screen and move the cursor to 1, 1
    printf("\033[?25l"); // Turn off visible cursor
    printf( "%c\033[0m", 0x0f ); // Reset all styles
    printf( "\033[32m" ); // bright green text
    printf("Self Test in Progress, Press Any Key to Exit"); // Added instruction to exit Matrix
    fflush(stdout); // send the codes to the terminal
    bool run = true;
// initialise trails
    for ( int i = 1; i < WIDTH; i++) {
        initTrail(&trails[i]);
    }
    // you are in the Matrix (TM)
    while(run) {
        if (serial_port.readable()) {
            serial_port.getc();
            run = false;
        }
        yellowLed = rand()%2;
        boilerLed = rand()%2;
        lightingLed = rand()%2;
        led = rand()%2;
        for (int x = 1; x < WIDTH; x++) {
            (trails[x].pos)++;
            if (trails[x].pos == HEIGHT) initTrail(&trails[x]);
        }
        printf("\033[H"); //Home
        for (int y = 1; y < HEIGHT; y++) {
            printf("\r\n");
            for (int x = 1; x < WIDTH; x++) {
                if ((trails[x].show < 2) || (trails[x].pos < y) || (trails[x].pos - 9 > y) ) printf( " " );
                else if (y <= trails[x].pos) {
                    switch (trails[x].pos - y) {
                        case 0:
                            printf( "\033[1;37m" );  //leader
                            break;
                        case 1:
                            printf ( "\033[1;32m" ); // follower
                            break;
                        case 2:
                            printf ( "\033[0;32m" ); // follower
                            break;
                        case 3:
                        case 4:
                        case 5:
                            printf( "\033[38;5;28m" ); // dim cell
                            break;
                        default:
                            printf( "\033[38;5;22m" ); // dimmest cell
                    }
                    if (trails[x].pos == trails[x].end) initTrail(&trails[x]);
                    printf("%c", trails[x].matStr[y]);
                }
            }
        }
    }
    run = true;
    yellowLed = myData.timeStatus; // BUG FIX: Stops LED being left ON/OFF when self test is exited
   
}
void initTrail( struct trail *trail )
{
    trail -> start = (rand() % (HEIGHT/2) + 1);
    trail -> end = trail -> start + 5 + rand()%(HEIGHT/2);
    trail -> speed = 1;
    trail -> pos = trail -> start;
    trail -> show = rand()%3;
    for (int i = 0; i < HEIGHT; i++) {
        if ( (i < trail -> start ) || (i > trail -> end )) trail->matStr[i] = 0x20;
        else trail->matStr[i] = 0x21 + abs(rand()%94);
    }

}