#ifndef __MMA8451Q_H__
#define __MMA8451Q_H__

#include "mbed.h"

/* Xtrinsic accelerometer */

/*
 * MMA8451 Registers
 */
#define MMA8451_STATUS                     0x00
#define MMA8451_OUT_X_MSB                  0x01
#define MMA8451_SYSMOD                     0x0b // 
#define MMA8451_INT_SOURCE                 0x0c // 
#define MMA8451_ID                         0x0d
#define MMA8451_PL_STATUS                  0x10
#define MMA8451_PL_CFG                     0x11
#define MMA8451_PL_COUNT                   0x12 // orientation debounce
#define MMA8451_PL_BF_ZCOMP                0x13
#define MMA8451_PL_THS_REG                 0x14
#define MMA8451_FF_MT_SRC                  0x16
#define MMA8451_TRANSIENT_CFG              0x1d // transient enable
#define MMA8451_TRANSIENT_SRC              0x1e // transient read/clear interrupt
#define MMA8451_TRANSIENT_THS              0x1f // transient threshold
#define MMA8451_TRANSIENT_COUNT            0x20 // transient debounce
#define MMA8451_PULSE_SRC                  0x22
#define MMA8451_CTRL_REG1                  0x2a
#define MMA8451_CTRL_REG2                  0x2b
#define MMA8451_CTRL_REG3                  0x2c // interrupt control
#define MMA8451_CTRL_REG4                  0x2d // interrupt enable
#define MMA8451_CTRL_REG5                  0x2e // interrupt pin selection


typedef union {
    struct {
        int16_t x;
        int16_t y;
        int16_t z;
    } v;
    uint8_t octets[6];
} mma_out_t;

typedef union { 
    struct {    // at 0x0c
        uint8_t SRC_DRDY    : 1;    // 0
        uint8_t reserved1   : 1;    // 1
        uint8_t SRC_FF_MT   : 1;    // 2
        uint8_t SRC_PULSE   : 1;    // 3
        uint8_t SRC_LNDPRT  : 1;    // 4
        uint8_t SRC_TRANS   : 1;    // 5
        uint8_t reserved6   : 1;    // 6
        uint8_t SRC_ASLP    : 1;    // 7
    } bits;
    uint8_t octet;
} mma_int_source_t;

typedef union { 
    struct {    // at 0x10
        uint8_t BAFRO  : 1;    // 0     0=front, 1=back
        uint8_t LAPO   : 2;    // 1,2   up, down, right, left
        uint8_t res    : 3;    // 3,4,5
        uint8_t LO     : 1;    // 6    Z-tilt lockout
        uint8_t NEWLP  : 1;    // 7    1 = BAFRO or LO has changed
    } bits;
    uint8_t octet;
} mma_pl_status_t;

typedef union {
    struct {    // at 0x1d
        uint8_t HPF_BYP : 1;    // 0
        uint8_t XTEFE   : 1;    // 1
        uint8_t YTEFE   : 1;    // 2
        uint8_t ZTEFE   : 1;    // 3
        uint8_t ELE     : 1;    // 4
        uint8_t pad     : 3;    // 5,6,7
    } bits;
    uint8_t octet;
} transient_cfg_t;

typedef union {
    struct {    // at 0x1e
        uint8_t X_Trans_Pol : 1;    // 0
        uint8_t XTRANSE     : 1;    // 1
        uint8_t Y_Trans_Pol : 1;    // 2
        uint8_t YTRANSE     : 1;    // 3
        uint8_t Z_Trans_Pol : 1;    // 4
        uint8_t ZTRANSE     : 1;    // 5
        uint8_t EA          : 1;    // 6
        uint8_t pad         : 1;    // 7
    } bits;
    uint8_t octet;
} transient_src_t;

typedef union { 
    struct {    // at 0x2a
        uint8_t ACTIVE      : 1;    // 0
        uint8_t F_READ      : 1;    // 1
        uint8_t LNOISE      : 1;    // 2
        uint8_t DR          : 3;    // 3,4,5
        uint8_t ASLP_RATE   : 2;    // 6,7
    } bits;
    uint8_t octet;
} ctrl_reg1_t;

typedef union { 
    struct {    // at 0x2d
        uint8_t INT_EN_DRDY     : 1;    // 0
        uint8_t reserved1       : 1;    // 1
        uint8_t INT_EN_FF_MT    : 1;    // 2
        uint8_t INT_EN_PULSE    : 1;    // 3
        uint8_t INT_EN_LNDPRT   : 1;    // 4
        uint8_t INT_EN_TRANS    : 1;    // 5
        uint8_t reserved6       : 1;    // 6
        uint8_t INT_EN_ASLP     : 1;    // 7
    } bits;
    uint8_t octet;
} ctrl_reg4_t;

typedef union { 
    struct {    // at 0x2e
        uint8_t INT_CFG_DRDY     : 1;    // 0
        uint8_t reserved1        : 1;    // 1
        uint8_t INT_CFG_FF_MT    : 1;    // 2
        uint8_t INT_CFG_PULSE    : 1;    // 3
        uint8_t INT_CFG_LNDPRT   : 1;    // 4
        uint8_t INT_CFG_TRANS    : 1;    // 5
        uint8_t reserved6        : 1;    // 6
        uint8_t INT_CFG_ASLP     : 1;    // 7
    } bits;
    uint8_t octet;
} ctrl_reg5_t;


struct MMA_orientation
{
    bool up;
    bool down;
    bool right;
    bool left;
    bool front;
    bool back;
    bool low;

    mma_pl_status_t pl_status;

    MMA_orientation()
    {
        reset();
    }

    inline void reset()
    {
        up=down=right=left=front=back=low=false;
        pl_status.octet = 0;
    }

    bool operator==(const MMA_orientation &rhs)
    {
        return ((up == rhs.up)       && 
                (down == rhs.down)   && 
                (left == rhs.left)   && 
                (front == rhs.front) && 
                (back == rhs.back)   && 
                (low == rhs.low));
    }

    bool operator!=(const MMA_orientation &rhs) { return !operator==(rhs); }

    
};

class MMA8451Q {
    public:
        MMA8451Q(I2C& r, DigitalIn& int_pin);
        ~MMA8451Q();
        void print_regs(void);
        void set_active(char);
        bool get_active(void);

        uint8_t read_single(uint8_t addr);
        void read(uint8_t addr, uint8_t *dst_buf, int length);
        void write(uint8_t addr, uint8_t data);
        void transient_detect(void);
        void orient_detect(void);
        uint8_t service(void); // returns 0 if no interrupt occurred

        inline MMA_orientation& getOrientation() { return orientation; }

        bool verbose;   // print interrupt event
        mma_out_t out;
        transient_cfg_t transient_cfg;
        ctrl_reg1_t ctrl_reg1;
        ctrl_reg4_t ctrl_reg4;
        ctrl_reg5_t ctrl_reg5;
        
    private:
        I2C& m_i2c;
        DigitalIn& m_int_pin;
        MMA_orientation orientation;
};

#endif
