#include "mbed.h"
#include "scheduler.h"
#include "LinkedList.h"
#include "jobService.h"
#include "schedules.h"

#include "mbed-trace/mbed_trace.h"
#define TRACE_GROUP  "main"

 // These are necessary only if thread safety is needed
static Mutex TracingLock;
static void tracingWait()
{
    TracingLock.lock();
}
static void tracingRelease()
{
    TracingLock.unlock();
}

void jobA(void *context)
{
    printf("Job A\n");
}

void jobB(void *context)
{
    printf("Job B\n");
}

void printJobs(JobScheduler::Scheduler& scheduler)
{
    LinkedList<JobScheduler::Job> jobs;
    scheduler.JobList(jobs);
    printf("  Job List:\n");
    node<JobScheduler::Job>* jn = jobs.pop(1);
    while (jn != NULL) {
        printf("Job ID: %d\n", jn->data->GetID());
        jobs.remove(1);
        jn = jobs.pop(1);
    }
    printf("  Job List Completed\n");
}


void runScheduler() 
{
    /* Setup tracing */
    mbed_trace_mutex_wait_function_set( tracingWait ); // only if thread safety is needed
    mbed_trace_mutex_release_function_set( tracingRelease ); // only if thread safety is needed
    mbed_trace_init();       // initialize the trace library

    JobScheduler::JobService js;
    JobScheduler::Scheduler scheduler(&js);
    
    JobScheduler::Error err;
    err = js.Register(3, NULL, NULL);
    if (err) {
        printf("Failed to register job type 1.\n");
        exit(1);
    }
    err = js.Register(1, jobA, NULL);
    if (err) {
        printf("Failed to register job type 1.\n");
        exit(1);
    }
    err = js.Register(2, jobB, NULL);
    if (err) {
        printf("Failed to register job type 2.\n");
        exit(1);
    }
    
    JobScheduler::JobType *jt;
    jt = js.GetJob(1);
    if (jt == NULL) {
        printf("Failed to find job type 1.\n");
        exit(1);
    }
    jt = js.GetJob(2);
    if (jt == NULL) {
        printf("Failed to find job type 2.\n");
        exit(1);
    }
    jt = js.GetJob(3);
    if (jt == NULL) {
        printf("Failed to find job type 2.\n");
        exit(1);
    }
    
    scheduler.Start();

    time_t nowSecs = time(NULL);
    JobScheduler::Response<JobScheduler::JobID> res = scheduler.JobAdd(1, new JobScheduler::RunOnceSchedule(nowSecs + 2), NULL);
    printf("job A add response error: %d, jobID: %d\n", res.error, res.data);
    
    printJobs(scheduler);
    wait(5);
    
    printf("periodic with 2 sec period and limit 3\n");
    res = scheduler.JobAdd(2, new JobScheduler::RunPeriodicSchedule(2, 3), NULL);
    printf("job B add response error: %d, jobID: %d\n", res.error, res.data);
    printJobs(scheduler);
    wait(10);
    printJobs(scheduler);
    scheduler.Stop();
    scheduler.WaitToStop();
}

int main()
{
    Serial pc(USBTX, USBRX);
    pc.baud(115200);

    printf("\nJob Scheduler Demo\n");
    runScheduler();
    printf("done\n");
 
    exit(0);
}

