/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "bmi160.h"


const struct BMI160::AccConfig BMI160::DEFAULT_ACC_CONFIG = {SENS_2G, ACC_US_OFF, ACC_BWP_2, ACC_ODR_8};
    

//*****************************************************************************
int32_t BMI160::setSensorPowerMode(Sensors sensor, PowerModes pwrMode)
{
    int32_t rtnVal = -1;
    
    switch(sensor)
    {
        case MAG:
            rtnVal = writeRegister(CMD, (MAG_SET_PMU_MODE | pwrMode));
        break;
        
        case GYRO:
            rtnVal = writeRegister(CMD, (GYR_SET_PMU_MODE | pwrMode));
        break;
        
        case ACC:
            rtnVal = writeRegister(CMD, (ACC_SET_PMU_MODE | pwrMode));
        break;
        
        default:
            rtnVal = -1;
        break;
    }
    
    return rtnVal;
}


//*****************************************************************************
int32_t BMI160::getTemperature(float *temp)
{
    uint8_t data[2];
    uint16_t rawTemp;
    
    int32_t rtnVal = readBlock(TEMPERATURE_0, TEMPERATURE_1, data);
    if(rtnVal == RTN_NO_ERROR)
    {
        rawTemp = ((data[1] << 8) | data[0]);
        if(rawTemp & 0x8000)
        {
            *temp = (23.0F - ((0x10000 - rawTemp)/512.0F));
        }
        else
        {
            *temp = ((rawTemp/512.0F) + 23.0F);
        }
    }
    
    return rtnVal;
}


//*****************************************************************************
int32_t BMI160::getSensorDataAndTime(uint8_t *data)
{
    return readBlock(DATA_0, SENSORTIME_2, data);
}


//*****************************************************************************
int32_t BMI160::getAccAxis(SensorAxis axis, AxisData &data, AccConfig accConfig)
{
    uint8_t localData[2];
    int32_t rtnVal;
    
    switch(axis)
    {
        case X_AXIS:
            rtnVal = readBlock(DATA_14, DATA_15, localData);
        break;
        
        case Y_AXIS:
            rtnVal = readBlock(DATA_16, DATA_17, localData);
        break;
        
        case Z_AXIS:
            rtnVal = readBlock(DATA_18, DATA_19, localData);
        break;
        
        default:
            rtnVal = -1;
        break;
    }
    
    if(rtnVal == RTN_NO_ERROR)
    {
        data.raw = ((localData[1] << 8) | localData[0]);
        switch(accConfig.range)
        {
            //magic numbers are typical values for LSB/g from EC table
            case SENS_2G:
                data.scaled = (data.raw/16384.0F);
            break;
            
            case SENS_4G:
                data.scaled = (data.raw/8192.0F);
            break;
            
            case SENS_8G:
                data.scaled = (data.raw/4096.0F);
            break;
            
            case SENS_16G:
                data.scaled = (data.raw/2048.0F);
            break;
        }
    }
    
    return rtnVal;
}
    

//*****************************************************************************    
int32_t BMI160::getAccXYZ(SensorData &data, AccConfig accConfig)
{
    uint8_t localData[6];
    int32_t rtnVal = readBlock(DATA_14, DATA_19, localData);
    
    if(rtnVal == RTN_NO_ERROR)
    {
        data.xAxis.raw = ((localData[1] << 8) | localData[0]);
        data.yAxis.raw = ((localData[3] << 8) | localData[2]);
        data.zAxis.raw = ((localData[5] << 8) | localData[4]);
        
        switch(accConfig.range)
        {
            //magic numbers are typical values for LSB/g from EC table
            case SENS_2G:
                data.xAxis.scaled = (data.xAxis.raw/16384.0F);
                data.yAxis.scaled = (data.yAxis.raw/16384.0F);
                data.zAxis.scaled = (data.zAxis.raw/16384.0F);
            break;
            
            case SENS_4G:
                data.xAxis.scaled = (data.xAxis.raw/8192.0F);
                data.yAxis.scaled = (data.yAxis.raw/8192.0F);
                data.zAxis.scaled = (data.zAxis.raw/8192.0F);
            break;
            
            case SENS_8G:
                data.xAxis.scaled = (data.xAxis.raw/4096.0F);
                data.yAxis.scaled = (data.yAxis.raw/4096.0F);
                data.zAxis.scaled = (data.zAxis.raw/4096.0F);
            break;
            
            case SENS_16G:
                data.xAxis.scaled = (data.xAxis.raw/2048.0F);
                data.yAxis.scaled = (data.yAxis.raw/2048.0F);
                data.zAxis.scaled = (data.zAxis.raw/2048.0F);
            break;
        }
    }
    
    return rtnVal;
}


//***************************************************************************** 
int32_t BMI160::getSensorTime(float *data)
{
    uint8_t localData[3];
    int32_t rtnVal = readBlock(SENSORTIME_0, SENSORTIME_2, localData);
    
    if(rtnVal == RTN_NO_ERROR)
    {
        *data = (((localData[2] << 16) | (localData[1] << 8) | localData[0]) * 39e-6);
    }
    
    return rtnVal;
}

