/*******************************************************************************
Copyright © 2016, STMicroelectronics International N.V.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
	* Redistributions of source code must retain the above copyright
	  notice, this list of conditions and the following disclaimer.
	* Redistributions in binary form must reproduce the above copyright
	  notice, this list of conditions and the following disclaimer in the
	  documentation and/or other materials provided with the distribution.
	* Neither the name of STMicroelectronics nor the
	  names of its contributors may be used to endorse or promote products
	  derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
NON-INFRINGEMENT OF INTELLECTUAL PROPERTY RIGHTS ARE DISCLAIMED.
IN NO EVENT SHALL STMICROELECTRONICS INTERNATIONAL N.V. BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/**
 * @file VL53L0X_def.h
 *
 * @brief Type definitions for VL53L0X API.
 *
 */


#ifndef _VL53L0X_DEF_H_
#define _VL53L0X_DEF_H_


#ifdef __cplusplus
extern "C" {
#endif

/** @defgroup VL53L0X_globaldefine_group VL53L0X Defines
 *	@brief	  VL53L0X Defines
 *	@{
 */

/****************** define for i2c configuration *******************************/
/** Maximum buffer size to be used in i2c */
#define VL53L0X_MAX_I2C_XFER_SIZE   64 /* Maximum buffer size to be used in i2c */
#define VL53L0X_I2C_USER_VAR         /* none but could be for a flag var to get/pass to mutex interruptible  return flags and try again */


/** Device SPECIFICATION major version */
#define VL53L0X10_SPECIFICATION_VER_MAJOR   1
/** Device SPECIFICATION minor version */
#define VL53L0X10_SPECIFICATION_VER_MINOR   2
/** Device SPECIFICATION sub version */
#define VL53L0X10_SPECIFICATION_VER_SUB	   7
/** Device SPECIFICATION sub version */
#define VL53L0X10_SPECIFICATION_VER_REVISION 1440

/** VL53L0X Device IMPLEMENTATION major version */
#define VL53L0X10_IMPLEMENTATION_VER_MAJOR	1
/** VL53L0X Device IMPLEMENTATION minor version */
#define VL53L0X10_IMPLEMENTATION_VER_MINOR	0
/** VL53L0X Device IMPLEMENTATION sub version */
#define VL53L0X10_IMPLEMENTATION_VER_SUB		9
/** VL53L0X Device IMPLEMENTATION sub version */
#define VL53L0X10_IMPLEMENTATION_VER_REVISION	3673

/** Device SPECIFICATION major version */
#define VL53L0X_SPECIFICATION_VER_MAJOR	 1
/** Device SPECIFICATION minor version */
#define VL53L0X_SPECIFICATION_VER_MINOR	 2
/** Device SPECIFICATION sub version */
#define VL53L0X_SPECIFICATION_VER_SUB	 7
/** Device SPECIFICATION sub version */
#define VL53L0X_SPECIFICATION_VER_REVISION 1440

/** VL53L0X Device IMPLEMENTATION major version */
#define VL53L0X_IMPLEMENTATION_VER_MAJOR	  1
/** VL53L0X Device IMPLEMENTATION minor version */
#define VL53L0X_IMPLEMENTATION_VER_MINOR	  1
/** VL53L0X Device IMPLEMENTATION sub version */
#define VL53L0X_IMPLEMENTATION_VER_SUB	  21
/** VL53L0X Device IMPLEMENTATION sub version */
#define VL53L0X_IMPLEMENTATION_VER_REVISION	  4823


#define VL53L0X_DEFAULT_MAX_LOOP 2000
#define VL53L0X_MAX_STRING_LENGTH 32

/**  The device model ID   */
#define IDENTIFICATION_MODEL_ID  0x000
#define STATUS_OK                0x00
#define STATUS_FAIL              0x01

/** default device address */
#define VL53L0X_DEFAULT_ADDRESS		0x52 /* (8-bit) */

#define  VL53L0X_STRING_DEVICE_INFO_NAME_TS0      "VL53L0X TS0"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_TS1      "VL53L0X TS1"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_TS2      "VL53L0X TS2"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_ES1      "VL53L0X ES1 or later"
#define  VL53L0X_STRING_DEVICE_INFO_TYPE          "VL53L0X"

#define REF_ARRAY_SPAD_0  0
#define REF_ARRAY_SPAD_5  5
#define REF_ARRAY_SPAD_10 10


/* sensor operating modes */
typedef enum {
    op_single_shot_poll = 1,
    op_poll,  // was range_continuous_polling
    op_INT,
    op_poll_low_threshold,
    op_poll_high_threshold,
    op_poll_out_of_window,
    op_INT_low_threshold,
    op_INT_high_threshold,
    op_INT_out_of_window,
} TOperatingMode;

/**
 * Device specific defines. To be adapted by implementer for the targeted
 * device.
 */

/** use where fractional values are expected
 *
 * Given a floating point value f it's 16.16 bit point is (int)(f*(1<<16))*/
typedef uint32_t TFP1616;

/** @defgroup VL53L0X_DevSpecDefines_group VL53L0X cut1.1 Device Specific Defines
 *  @brief VL53L0X cut1.1 Device Specific Defines
 *  @{
 */

/** @defgroup VL53L0X_DeviceError_group Device Error
 *  @brief Device Error code
 *
 *  This enum is Device specific it should be updated in the implementation
 *  Use @a VL53L0X_GetStatusErrorString() to get the string.
 *  It is related to Status Register of the Device.
 *  @{
 */
typedef uint8_t TDevError;

#define VL53L0X_DEVICEERROR_NONE                        ((TDevError) 0)
/*!< 0  NoError  */
#define VL53L0X_DEVICEERROR_VCSELCONTINUITYTESTFAILURE  ((TDevError) 1)
#define VL53L0X_DEVICEERROR_VCSELWATCHDOGTESTFAILURE    ((TDevError) 2)
#define VL53L0X_DEVICEERROR_NOVHVVALUEFOUND             ((TDevError) 3)
#define VL53L0X_DEVICEERROR_MSRCNOTARGET                ((TDevError) 4)
#define VL53L0X_DEVICEERROR_SNRCHECK                    ((TDevError) 5)
#define VL53L0X_DEVICEERROR_RANGEPHASECHECK             ((TDevError) 6)
#define VL53L0X_DEVICEERROR_SIGMATHRESHOLDCHECK         ((TDevError) 7)
#define VL53L0X_DEVICEERROR_TCC                         ((TDevError) 8)
#define VL53L0X_DEVICEERROR_MINCLIP                     ((TDevError) 10)
#define VL53L0X_DEVICEERROR_RANGECOMPLETE               ((TDevError) 11)
#define VL53L0X_DEVICEERROR_ALGOUNDERFLOW               ((TDevError) 12)
#define VL53L0X_DEVICEERROR_ALGOOVERFLOW                ((TDevError) 13)
#define VL53L0X_DEVICEERROR_RANGEIGNORETHRESHOLD        ((TDevError) 14)

/** @} end of VL53L0X_DeviceError_group */


/** @defgroup VL53L0X_CHECKEN_group Check Enable list
 *  @brief Check Enable code
 *
 *  Define used to specify the LimitCheckId.
 *  Use @a VL53L0X_GetLimitCheckInfo() to get the string.
 *  @{
 */
#define VL53L0X_CHECKEN_SIGMA_FINAL_RANGE        0
#define VL53L0X_CHECKEN_SIG_RATE_FINAL_RANGE     1
#define VL53L0X_CHECKEN_SIG_REF_CLIP             2
#define VL53L0X_CHECKEN_RANGE_IGNORE_THRESHOLD   3
#define VL53L0X_CHECKEN_SIG_RATE_MSRC            4
#define VL53L0X_CHECKEN_SIG_RATE_PRE_RANGE       5
#define VL53L0X_CHECKEN_NUMBER_OF_CHECKS         6
/** @}  end of VL53L0X_CHECKEN_group */


/** @defgroup GPIO_Func_group Gpio Functionality
 *  @brief Defines the different functionalities for the device GPIO(s)
 *  @{   */
typedef enum {
    GPIO_FUNC_OFF = 0, /*!< NO Interrupt  */
    GPIO_FUNC_THRESHOLD_CROSSED_LOW,  /*!< Level Low (value < thresh_low)  */
    GPIO_FUNC_THRESHOLD_CROSSED_HIGH, /*!< Level High (value > thresh_high) */
    GPIO_FUNC_THRESHOLD_CROSSED_OUT, /*!< Out Of Window (value < thresh_low OR value > thresh_high)  */
    GPIO_FUNC_NEW_MEASURE_READY /*!< New Sample Ready  */
} TGPIO_Func;
/** @} end of GPIO_FUNC_group */


/* Device register map */

/** @defgroup VL53L0X_DefineRegisters_group Define Registers
 *  @brief List of all the defined registers
 *  @{
 */
#define REG_SYSRANGE_START              0x000
/** mask existing bit in #REG_SYSRANGE_START*/
#define REG_SYSRANGE_MODE_MASK          0x0F
/** bit 0 in #REG_SYSRANGE_START write 1 toggle state in
 * continuous mode and arm next shot in single shot mode */
#define REG_SYSRANGE_MODE_START_STOP    0x01
/** bit 1 write 0 in #REG_SYSRANGE_START set single shot mode */
#define REG_SYSRANGE_MODE_SINGLESHOT    0x00
/** bit 1 write 1 in #REG_SYSRANGE_START set back-to-back
 *  operation mode */
#define REG_SYSRANGE_MODE_BACKTOBACK    0x02
/** bit 2 write 1 in #REG_SYSRANGE_START set timed operation
 *  mode */
#define REG_SYSRANGE_MODE_TIMED         0x04
/** bit 3 write 1 in #REG_SYSRANGE_START set histogram operation
 *  mode */
#define REG_SYSRANGE_MODE_HISTOGRAM     0x08

#define REG_SYSTEM_THRESH_HIGH          0x0C
#define REG_SYSTEM_THRESH_LOW           0x0E
#define REG_SYSTEM_SEQUENCE_CONFIG		0x01
#define REG_SYSTEM_RANGE_CONFIG			0x09
#define REG_SYSTEM_MEASURE_PERIOD		0x04

#define REG_SYSINT_CONFIG_GPIO          0x0A
#define REG_SYSINT_GPIO_DISABLED		0x00
#define REG_SYSINT_GPIO_LEVEL_LOW		0x01
#define REG_SYSINT_GPIO_LEVEL_HIGH		0x02
#define REG_SYSINT_GPIO_OUT_OF_WINDOW	0x03
#define REG_SYSINT_GPIO_NEW_SAMPLE_READY 0x04

#define REG_GPIO_HV_MUX_ACTIVE_HIGH     0x84
#define REG_SYSINT_CLEAR           		0x0B

/* Result registers */
#define REG_RESULT_INTERRUPT_STATUS     0x13
#define REG_RESULT_RANGE_STATUS         0x14

#define REG_RESULT_CORE_PAGE  1
#define REG_RESULT_CORE_AMBIENT_WINDOW_EVENTS_RTN   0xBC
#define REG_RESULT_CORE_RANGING_TOTAL_EVENTS_RTN    0xC0
#define REG_RESULT_CORE_AMBIENT_WINDOW_EVENTS_REF   0xD0
#define REG_RESULT_CORE_RANGING_TOTAL_EVENTS_REF    0xD4
#define REG_RESULT_PEAK_SIG_RATE_REF                0xB6

/* Algo register */
#define REG_ALGO_PART_TO_PART_RANGE_OFFSET_MM       0x28
#define REG_I2C_SLAVE_DEVICE_ADDRESS                0x8a

/* Check Limit registers */
#define REG_MSRC_CONFIG_CONTROL                     0x60

#define REG_PRE_RANGE_CONFIG_MIN_SNR                0X27
#define REG_PRE_RANGE_CONFIG_VALID_PHASE_LOW        0x56
#define REG_PRE_RANGE_CONFIG_VALID_PHASE_HIGH       0x57
#define REG_PRE_RANGE_MIN_COUNT_RATE_RTN_LIMIT      0x64

#define REG_FINAL_RANGE_CONFIG_MIN_SNR              0X67
#define REG_FINAL_RANGE_CONFIG_VALID_PHASE_LOW      0x47
#define REG_FINAL_RANGE_CONFIG_VALID_PHASE_HIGH     0x48
#define REG_FINAL_RANGE_CONFIG_MIN_COUNT_RATE_RTN_LIMIT   0x44

#define REG_PRE_RANGE_CONFIG_SIGMA_THRESH_HI        0x61
#define REG_PRE_RANGE_CONFIG_SIGMA_THRESH_LO        0x62

/* PRE RANGE registers */
#define REG_PRE_RANGE_CONFIG_VCSEL_PERIOD           0x50
#define REG_PRE_RANGE_CONFIG_TIMEOUT_MACROP_HI      0x51
#define REG_PRE_RANGE_CONFIG_TIMEOUT_MACROP_LO      0x52

#define REG_SYSTEM_HISTOGRAM_BIN                    0x81
#define REG_HISTOGRAM_CONFIG_INITIAL_PHASE_SELECT   0x33
#define REG_HISTOGRAM_CONFIG_READOUT_CTRL           0x55

#define REG_FINAL_RANGE_CONFIG_VCSEL_PERIOD         0x70
#define REG_FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI    0x71
#define REG_FINAL_RANGE_CONFIG_TIMEOUT_MACROP_LO    0x72
#define REG_XTALK_COMPENS_RATE_MHz                  0x20

#define REG_MSRC_CONFIG_TIMEOUT_MACROP              0x46

#define REG_SOFT_RESET_GO2_SOFT_RESET_N	            0xbf
#define REG_IDENTIFICATION_MODEL_ID                 0xc0
#define REG_IDENTIFICATION_REVISION_ID              0xc2

#define REG_OSC_CALIBRATE_VAL                       0xf8

#define VL53L0X_SIGMA_ESTIMATE_MAX_VALUE                    0xffff
/* equivalent to a range sigma of 655.35mm */

#define REG_GLOBAL_CONFIG_VCSEL_WIDTH          		0x32
#define REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_0   		0xB0
#define REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_1   		0xB1
#define REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_2   		0xB2
#define REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_3   		0xB3
#define REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_4   		0xB4
#define REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_5   		0xB5

#define REG_GLOBAL_CONFIG_REF_EN_START_SELECT   	0xB6
#define REG_DYNAMIC_SPAD_NUM_REQUESTED_REF_SPAD 	0x4E
#define REG_DYNAMIC_SPAD_REF_EN_START_OFFSET    	0x4F
#define REG_POWER_MANAGEMENT_GO1_POWER_FORCE    	0x80

/*  Speed of light in um per 1E-10 Seconds */
#define VL53L0X_SPEED_OF_LIGHT_IN_AIR 2997

#define REG_VHV_CONFIG_PAD_SCL_SDA__EXTSUP_HV   0x89

#define REG_ALGO_PHASECAL_LIM                   0x30
#define REG_ALGO_PHASECAL_CONFIG_TIMEOUT        0x30

/** @} VL53L0X_DefineRegisters_group */

/** @} VL53L0X_DevSpecDefines_group */


/****************************************
 * PRIVATE define do not edit
 ****************************************/

/** @brief Defines the parameters of the Get Version Functions
 */
typedef struct {
    uint32_t	 revision; /*!< revision number */
    uint8_t		 major;	   /*!< major number */
    uint8_t		 minor;	   /*!< minor number */
    uint8_t		 build;	   /*!< build number */
} VL53L0X_Version_t;


/** @brief Defines the parameters of the Get Device Info Functions
 */
typedef struct {
    char Name[VL53L0X_MAX_STRING_LENGTH];
    /*!< Name of the Device e.g. Left_Distance */
    char Type[VL53L0X_MAX_STRING_LENGTH];
    /*!< Type of the Device e.g VL53L0X */
    char ProductId[VL53L0X_MAX_STRING_LENGTH];
    /*!< Product Identifier String	*/
    uint8_t ProductType;
    /*!< Product Type, VL53L0X = 1, VL53L1 = 2 */
    uint8_t ProductRevisionMajor;
    /*!< Product revision major */
    uint8_t ProductRevisionMinor;
    /*!< Product revision minor */
} TVL53L0X_DeviceInfo;


typedef  TVL53L0X_DeviceInfo* PVL53L0X_DeviceInfo;

/** @defgroup VL53L0X_define_Error_group Error and Warning code returned by API
 *	The following DEFINE are used to identify the Device ERROR
 *	@{
 */

typedef int8_t VL53L0X_Error;

#define VL53L0X_OK					((VL53L0X_Error)  0)
#define VL53L0X_ERROR_CALIBRATION_WARNING	((VL53L0X_Error) -1)
/*!< Warning invalid calibration data may be in used
	\a	VL53L0X_InitData()
	\a VL53L0X_GetOffsetCalibrationData
	\a VL53L0X_SetOffsetCalibrationData */
#define VL53L0X_ERROR_MIN_CLIPPED			((VL53L0X_Error) -2)
/*!< Warning parameter passed was clipped to min before to be applied */
#define VL53L0X_ERROR_UNDEFINED				((VL53L0X_Error) -3)
/*!< Unqualified error */
#define VL53L0X_ERROR_INVALID_PARAMS		((VL53L0X_Error) -4)
/*!< Parameter passed is invalid or out of range */
#define VL53L0X_ERROR_NOT_SUPPORTED			((VL53L0X_Error) -5)
/*!< Function is not supported in current mode or configuration */
#define VL53L0X_ERROR_RANGE_ERROR			((VL53L0X_Error) -6)
/*!< Device report a ranging error interrupt status */
#define VL53L0X_ERROR_TIME_OUT				((VL53L0X_Error) -7)
/*!< Aborted due to time out */
#define VL53L0X_ERROR_MODE_NOT_SUPPORTED	((VL53L0X_Error) -8)
/*!< Asked mode is not supported by the device */
#define VL53L0X_ERROR_BUFFER_TOO_SMALL		((VL53L0X_Error) -9)
/*!< ... */
#define VL53L0X_ERROR_GPIO_NOT_EXISTING		((VL53L0X_Error) -10)
/*!< User tried to setup a non-existing GPIO pin */
#define VL53L0X_ERROR_GPIO_FUNC_NOT_SUPPORTED ((VL53L0X_Error) -11)
/*!< unsupported GPIO functionality */
#define VL53L0X_ERROR_INTERRUPT_NOT_CLEARED	((VL53L0X_Error) -12)
/*!< Error during interrupt clear */
#define VL53L0X_ERROR_CONTROL_INTERFACE		((VL53L0X_Error) -20)
#define VL53L0X_ERROR_I2C_BUF_OVERFLOW		((VL53L0X_Error) -22)
#define VL53L0X_ERROR_I2C_WRONG_DEV_ID		((VL53L0X_Error) -23)
/*!< error reported from IO functions */
#define VL53L0X_ERROR_INVALID_COMMAND		((VL53L0X_Error) -30)
/*!< The command is not allowed in the current device state (power down) */
#define VL53L0X_ERROR_DIVISION_BY_ZERO		((VL53L0X_Error) -40)
/*!< In the function a division by zero occurs */
#define VL53L0X_ERROR_REF_SPAD_INIT			((VL53L0X_Error) -50)
/*!< Error during reference SPAD initialization */
#define VL53L0X_ERROR_NOT_IMPLEMENTED		((VL53L0X_Error) -99)
/*!< Tells requested functionality has not been implemented yet or
 * not compatible with the device */
/** @} VL53L0X_define_Error_group */


/** @defgroup VL53L0X_define_DeviceModes_group Defines Device modes
 *	Defines all possible modes for the device
 *	@{
 */
typedef uint8_t VL53L0X_DeviceModes;

#define VL53L0X_DEVICEMODE_SINGLE_RANGING	((VL53L0X_DeviceModes)  0)
#define VL53L0X_DEVICEMODE_CONTINUOUS_RANGING	((VL53L0X_DeviceModes)  1)
#define VL53L0X_DEVICEMODE_SINGLE_HISTOGRAM	((VL53L0X_DeviceModes)  2)
#define VL53L0X_DEVICEMODE_CONTINUOUS_TIMED_RANGING ((VL53L0X_DeviceModes) 3)
#define VL53L0X_DEVICEMODE_SINGLE_ALS		((VL53L0X_DeviceModes) 10)
#define VL53L0X_DEVICEMODE_GPIO_DRIVE		((VL53L0X_DeviceModes) 20)
#define VL53L0X_DEVICEMODE_GPIO_OSC		((VL53L0X_DeviceModes) 21)
/* ... Modes to be added depending on device */
/** @} VL53L0X_define_DeviceModes_group */

/** @defgroup VL53L0X_define_HistogramModes_group Defines Histogram modes
 *	Defines all possible Histogram modes for the device
 *	@{
 */
typedef uint8_t VL53L0X_HistogramModes;

#define VL53L0X_HISTOGRAMMODE_DISABLED		((VL53L0X_HistogramModes) 0)
/*!< Histogram Disabled */
#define VL53L0X_HISTOGRAMMODE_REFERENCE_ONLY	((VL53L0X_HistogramModes) 1)
/*!< Histogram Reference array only */
#define VL53L0X_HISTOGRAMMODE_RETURN_ONLY	((VL53L0X_HistogramModes) 2)
/*!< Histogram Return array only */
#define VL53L0X_HISTOGRAMMODE_BOTH		((VL53L0X_HistogramModes) 3)
/*!< Histogram both Reference and Return Arrays */
/* ... Modes to be added depending on device */
/** @} VL53L0X_define_HistogramModes_group */


/** @defgroup VL53L0X_define_PowerModes_group List of available Power Modes
 *	List of available Power Modes
 *	@{
 */

typedef uint8_t VL53L0X_PowerModes;

#define VL53L0X_POWERMODE_STANDBY_LEVEL1 ((VL53L0X_PowerModes) 0)
/*!< Standby level 1 */
#define VL53L0X_POWERMODE_STANDBY_LEVEL2 ((VL53L0X_PowerModes) 1)
/*!< Standby level 2 */
#define VL53L0X_POWERMODE_IDLE_LEVEL1	((VL53L0X_PowerModes) 2)
/*!< Idle level 1 */
#define VL53L0X_POWERMODE_IDLE_LEVEL2	((VL53L0X_PowerModes) 3)
/*!< Idle level 2 */

/** @} VL53L0X_define_PowerModes_group */

/** @brief Defines all parameters for the device
 */
typedef struct {
    VL53L0X_DeviceModes DeviceMode;
    /*!< Defines type of measurement to be done for the next measure */
    VL53L0X_HistogramModes HistogramMode;
    /*!< Defines type of histogram measurement to be done for the next
     *	measure */
    uint32_t Measure_Time_Budget_us; // was MeasurementTimingBudget_us
    /*!< Defines the allowed total time for a single measurement */
    uint32_t Measure_Period_ms;
    /*!< Defines time between two consecutive measurements (between two
     *	measurement starts). If set to 0 means back-to-back mode */
    uint8_t XTalk_Compens_En;
    /*!< Tells if Crosstalk compensation shall be enable or not	 */
    uint16_t XTalkCompensationRangeMilliMeter;
    /*!< CrossTalk compensation range in millimeter	 */
    TFP1616 Xtalk_CompRate_MHz;
    /*!< CrossTalk compensation rate in Mega counts per seconds.
     *	Expressed in 16.16 fixed point format.	*/
    int32_t Offset_Cal_um;
    /*!< Range offset adjustment (um) last programmed.	*/

    uint8_t Limit_Chk_En[VL53L0X_CHECKEN_NUMBER_OF_CHECKS];
    /*!< This Array store all the Limit Check enable for this device. */
    uint8_t LimitChecksStatus[VL53L0X_CHECKEN_NUMBER_OF_CHECKS];
    /*!< This Array store all the Status of the check linked to last
    * measurement. */
    TFP1616 Limit_Chk_Val[VL53L0X_CHECKEN_NUMBER_OF_CHECKS];
    /*!< This Array store all the Limit Check value for this device */
    uint8_t Wrap_Around_Chk_En;
    /*!< Tells if Wrap Around Check shall be enable or not */
} VL53L0X_DeviceParams_t;


/** @defgroup VL53L0X_define_State_group Defines the current status of the device
 *	Defines the current status of the device
 *	@{
 */
typedef uint8_t VL53L0X_State;

#define VL53L0X_STATE_POWERDOWN		  ((VL53L0X_State)  0)
/*!< Device is in HW reset	*/
#define VL53L0X_STATE_WAIT_STATICINIT ((VL53L0X_State)  1)
/*!< Device is initialized and wait for static initialization  */
#define VL53L0X_STATE_STANDBY		  ((VL53L0X_State)  2)
/*!< Device is in Low power Standby mode   */
#define VL53L0X_STATE_IDLE			  ((VL53L0X_State)  3)
/*!< Device has been initialized and ready to do measurements  */
#define VL53L0X_STATE_RUNNING		  ((VL53L0X_State)  4)
/*!< Device is performing measurement */
#define VL53L0X_STATE_UNKNOWN		  ((VL53L0X_State)  98)
/*!< Device is in unknown state and need to be rebooted	 */
#define VL53L0X_STATE_ERROR			  ((VL53L0X_State)  99)
/*!< Device is in error state and need to be rebooted  */
/** @} VL53L0X_define_State_group */


/** @brief Structure containing the Dmax computation parameters and data
 */
typedef struct {
    int32_t AmbTuningWindowFactor_K;
    /*!<  internal algo tuning (*1000) */
    int32_t RetSignalAt0mm;
    /*!< intermediate dmax computation value caching */
} VL53L0X_DMaxData_t;

/**
 * @struct VL53L0X_RangeData_t
 * @brief Range measurement data.
 */
typedef struct {
    uint32_t TimeStamp;		/*!< 32-bit time stamp. */
    uint32_t MeasurementTimeUsec;
    /*!< Give the Measurement time needed by the device to do the measurement.*/

    uint16_t RangeMilliMeter;	/*!< range distance in millimeter. */

    uint16_t RangeDMaxMilliMeter;
    /*!< Tells what is the maximum detection distance of the device
     * in current setup and environment conditions (Filled when
     *	applicable) */

    TFP1616 SignalRateRtnMHz;
    /*!< Return signal rate (MHz)\n these is a 16.16 fix point
     *	value, which is effectively a measure of target
     *	 reflectance.*/
    TFP1616 AmbientRateRtnMHz;
    /*!< Return ambient rate (MHz)\n these is a 16.16 fix point
     *	value, which is effectively a measure of the ambien
     *	t light.*/

    uint16_t EffectiveSPADRtnCount;
    /*!< Return the effective SPAD count for the return signal.
     *	To obtain Real value it should be divided by 256 */

    uint8_t ZoneId;
    /*!< Denotes which zone and range scheduler stage the range
     *	data relates to. */
    uint8_t RangeFractionalPart;
    /*!< Fractional part of range distance. Final value is a
     *	FixPoint168 value. */
    uint8_t RangeStatus;
    /*!< Range Status for the current measurement. This is device
     *	dependent. Value = 0 means value is valid.
     *	See \ref RangeStatusPage */
} TRangeResults; // was VL53L0X_RangingMeasurementData_t


#define VL53L0X_HISTOGRAM_BUFFER_SIZE 24

/**
 * @struct VL53L0X_HistogramData_t
 * @brief Histogram measurement data.
 */
typedef struct {  /* Histogram Measurement data */
    uint32_t HistogramData[VL53L0X_HISTOGRAM_BUFFER_SIZE];
    /*!< Histogram data */
    uint8_t HistogramType; /*!< Indicate the types of histogram data :
	Return only, Reference only, both Return and Reference */
    uint8_t FirstBin; /*!< First Bin value */
    uint8_t BufferSize; /*!< Buffer Size - Set by the user.*/
    uint8_t NumberOfBins;/*!< Number of bins filled by the histogram measurement */
    TDevError ErrorStatus;
    /*!< Error status of the current measurement. \n
    see @a ::TDevError @a VL53L0X_GetStatusErrorString() */
} VL53L0X_HistogramMeasurementData_t;

#define VL53L0X_PLL_PERIOD_PS = 1655;
#define VL53L0X_MACRO_PERIOD_VCLKS = 2304;

/**
 * @struct VL53L0X_SPADData_t
 * @brief SPAD Configuration Data.
 */
 #define REF_SPAD_ARRAY_SIZE 6

typedef struct {
    uint8_t RefSPADEnables[REF_SPAD_ARRAY_SIZE];
    /*!< Reference SPAD Enables */
    uint8_t RefGoodSPADMap[REF_SPAD_ARRAY_SIZE];
    /*!< Reference SPAD Good SPAD Map */
} VL53L0X_SPADData_t;


/** @defgroup VL53L0X_define_InterruptPolarity_group Defines the Polarity
 * of the Interrupt
 *	Defines the Polarity of the Interrupt
 *	@{
 */
typedef uint8_t VL53L0X_InterruptPolarity;
#define VL53L0X_INTERRUPTPOLARITY_LOW	   ((VL53L0X_InterruptPolarity)	0)
/*!< Set active low polarity best setup for falling edge. */
#define VL53L0X_INTERRUPTPOLARITY_HIGH	   ((VL53L0X_InterruptPolarity)	1)
/*!< Set active high polarity best setup for rising edge. */

/** @} VL53L0X_define_InterruptPolarity_group */


/** @defgroup VL53L0X_define_VcselPeriod_group Vcsel Period Defines
 *	Defines the range measurement for which to access the vcsel Phase.
 *	@{
 */
typedef uint8_t VL53L0X_Range_Phase;

#define VL53L0X_VCSEL_PRE_RANGE	 ((VL53L0X_Range_Phase) 0)
/*!<Identifies the pre-range vcsel Phase. */
#define VL53L0X_VCSEL_FINAL_RANGE ((VL53L0X_Range_Phase) 1)
/*!<Identifies the final range vcsel Phase . */

/** @} VL53L0X_define_VcselPeriod_group */

/** @defgroup VL53L0X_define_SchedulerSequence_group Defines the steps
 * carried out by the scheduler during a range measurement.
 *	@{
 *	Defines the states of all the steps in the scheduler
 *	i.e. enabled/disabled.
 */
typedef struct {
    uint8_t		 TccOn;	   /*!<Reports if Target Centre Check On  */
    uint8_t		 MsrcOn;	   /*!<Reports if MSRC On  */
    uint8_t		 DssOn;		   /*!<Reports if DSS On  */
    uint8_t		 PreRangeOn;   /*!<Reports if Pre-Range On	*/
    uint8_t		 FinalRangeOn; /*!<Reports if Final-Range On  */
} VL53L0X_Sequence_Steps_t;

/** @} VL53L0X_define_SchedulerSequence_group */

/** @defgroup VL53L0X_define_SequenceStepId_group Defines the Polarity
 *	of the Interrupt
 *	Defines the the sequence steps performed during ranging..
 *	@{
 */
typedef uint8_t VL53L0X_SequenceStepId;

#define	 VL53L0X_SEQUENCESTEP_TCC		 ((VL53L0X_SequenceStepId) 0)
/*!<Target CentreCheck identifier. */
#define	 VL53L0X_SEQUENCESTEP_DSS		 ((VL53L0X_SequenceStepId) 1)
/*!<Dynamic SPAD Selection function Identifier. */
#define	 VL53L0X_SEQUENCESTEP_MSRC		 ((VL53L0X_SequenceStepId) 2)
/*!<Minimum Signal Rate Check function Identifier. */
#define	 VL53L0X_SEQUENCESTEP_PRE_RANGE	 ((VL53L0X_SequenceStepId) 3)
/*!<Pre-Range check Identifier. */
#define	 VL53L0X_SEQUENCESTEP_FINAL_RANGE ((VL53L0X_SequenceStepId) 4)
/*!<Final Range Check Identifier. */

#define	 VL53L0X_SEQUENCESTEP_NUMBER_OF_CHECKS			 5
/*!<Number of Sequence Step Managed by the API. */

/** @} VL53L0X_define_SequenceStepId_group */

/* MACRO Definitions */
/** @defgroup VL53L0X_define_GeneralMacro_group General Macro Defines
 *	General Macro Defines
 *	@{
 */

/* Defines */
#define FP1616_TO_FP97(Value) (uint16_t)((Value>>9)&0xFFFF)
#define FP97_TO_FP1616(Value) (TFP1616)(Value<<9)

#define FP1616_TO_FP88(Value) (uint16_t)((Value>>8)&0xFFFF)
#define FP88_TO_FP1616(Value) (TFP1616)(Value<<8)

#define FP1616_TO_FP412(Value) (uint16_t)((Value>>4)&0xFFFF)
#define FP412_TO_FP1616(Value) (TFP1616)(Value<<4)

#define FP1616_TO_FP313(Value) (uint16_t)((Value>>3)&0xFFFF)
#define FP313_TO_FP1616(Value) (TFP1616)(Value<<3)

#define FP1616_TO_FP08(Value) (uint8_t)((Value>>8)&0x00FF)
#define FP08_TO_FP1616(Value) (TFP1616)(Value<<8)

#define FP1616_TO_FP53(Value) (uint8_t)((Value>>13)&0x00FF)
#define FP53_TO_FP1616(Value) (TFP1616)(Value<<13)

#define FP1616_TO_FP102(Value) (uint16_t)((Value>>14)&0x0FFF)
#define FP102_TO_FP1616(Value) (TFP1616)(Value<<12)

#define VL53L0X_MAKEUINT16(lsb, msb) (uint16_t)((((uint16_t)msb)<<8) + (uint16_t)lsb)

/** @} VL53L0X_define_GeneralMacro_group */

/** @} VL53L0X_globaldefine_group */

#ifdef __cplusplus
}
#endif

#endif /* _VL53L0X_DEF_H_ */
