/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Tarek Lule, based on code by CLab
 * @version V1.0.0
 * @date    13-March-2019
 * @brief   Simple Example application for using TOF  Sensors
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2017 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 

/* Includes */
#include "mbed.h"
#include "VL53L0X.h"

int Ionisch  [] = {0, 2, 4, 5, 7, 9, 11};
int Dorisch  [] = {0, 2, 3, 5, 7, 9, 10};
int Phrygisch[] = {0, 1, 3, 5, 7, 8, 10};
int Lydisch  [] = {0, 2, 4, 6, 7, 9, 11};
int Mixolydi [] = {0, 2, 4, 5, 7, 9, 10};
int Eolisch  [] = {0, 2, 3, 5, 7, 8, 10};
int Lokrisch [] = {0, 1, 3, 5, 6, 8, 10};
int PhrygDomi[] = {0, 1, 4, 5, 7, 8, 10};
int MiShebera[] = {0, 2, 3, 6, 7, 9, 10};
int TsigaDur [] = {0, 2, 3, 6, 7, 8, 11}; // https://de.wikipedia.org/wiki/Zigeunertonleiter
int TsigaMoll[] = {0, 2, 3, 5, 7, 8, 11}; // https://de.wikipedia.org/wiki/Zigeunertonleiter

// Maquam: arabische leiter mit 3/4 Tonschritten, kann nur über CV Ausang funzen
// siehe: https://de.wikipedia.org/wiki/Maqam_(Musik)

/* Interface definition */
static DevI2C devI2c(PB_11,PB_10);
Serial midi_uart(PA_0, PA_1);  // PA_0 = Tx, PA_1 = Rx

/* Range sensor - B-L475E-IOT01A2 only */
static DigitalOut shutdown_pin(PC_6);
static VL53L0X tof1(&devI2c, &shutdown_pin, PC_7);

/* Simple main function */
int main() {
  uint32_t distance, NewDist, OldDist;
  int status;
  char NoteNr, Octave, ScaleNote;
  char OldKey, NewKey;
  VL53L0X_RangingMeasurementData_t p_ranging_measurement_data;
    
  /* Init all sensors with default params */
  tof1.init_sensor(VL53L0X_DEFAULT_ADDRESS);
  status = tof1.start_measurement(range_continuous_polling, NULL);
  status = status | tof1.VL53L0X_set_measurement_timing_budget_us(100000);
  if (status) 
     {  printf("\n\r--- Error Initializing the TOF sensor ---\n\r"); 
        //while(1) { }; // consider to hang up the device here
     }
  printf("\n\r--- Read Init of TOF ---\n\r"); ;
         
  OldKey = 0x00; // Remembers the old key, start with value thats normally unused 
  midi_uart.baud( 31250 );
  midi_uart.format( 8, SerialBase::None, 2);
  wait(0.1); // make sure all is settled
  midi_uart.putc(0xB0);  // Midi Channel 0, Channel Mode Message
  midi_uart.putc(123);   // Byte 2: All Notes Off
  midi_uart.putc(0x00);  // Byte 3: obligatory zero 
  
 
  while(1)
    { status =  tof1.get_measurement(range_continuous_polling, &p_ranging_measurement_data);
        
    if ( (status == VL53L0X_ERROR_NONE) && (p_ranging_measurement_data.RangeStatus == 0) )
        { // we have a valid range.
        NewDist = p_ranging_measurement_data.RangeMilliMeter;
        // printf("%6ld\r", distance); // Distance is ~1700 when looking at the ceiling, so approximately 1 per mm. A really usable distance range is only up to 1270. 
        distance = (int) ( NewDist / 30 + OldDist / 20 ) + 20; // scaling factor 10 between distance and key note
        if (distance >= 0x7F) NoteNr = 0x00; // cap the New Key Value on the top, beyond 127 we discard the note as 'no Note'
            else NoteNr = distance; // pass through everything below; remember: distance is uint, so no negatives can occur
        Octave = (int) (NoteNr / 7);
        ScaleNote = NoteNr - (Octave * 7);
        NewKey = Octave * 12 + TsigaMoll[ScaleNote];
        OldDist = NewDist;
        } 
      else 
        { // printf("  --\r");
        NewKey = 0x00;
        }
    if ( OldKey!=NewKey )
      {
        if (OldKey != 0x00) 
            { // turn off the old note first, if there was one
            midi_uart.putc(0x80);   // Midi Channel 0, Note OFF, 
            midi_uart.putc(OldKey); // Byte 2: Old Key Number [0 - 127]   
            midi_uart.putc(0x3F);   // Byte 3: Velocity [0 - 127]; here fixed=63 
            }
        if (NewKey != 0x00)
            { // turn on the new note now, if there is a valid one
            printf("%6ld\r\n", NewKey);
            midi_uart.putc(0x90);  // Midi Channel 0, Note ON, 
            midi_uart.putc(NewKey);// Byte 2: New Key Number [0 - 127]   
            midi_uart.putc(0x3F);  // Byte 3: Velocity [0 - 127]; here fixed=63 
            wait(0.1); // leave a little pause before a new note gets played ...
            }
        OldKey = NewKey;
        // wait(0.1);
      } // if ( OldKey!=NewKey )
  } // while(1)
} // int main()
