/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef RWD_MODULE_H
#define RWD_MODULE_H

#include "mbed.h"

//!Generic driver for the RWD RFID Modules from IB Technology.
/*!
The RWD modules from IB Technology are RFID readers working with different frequencies and protocols but with a common instructions set and pinout.
*/
class RWDModule
{
public:
  //!Creates an instance of the class.
  /*!
  Connect module using serial port pins tx, rx and DigitalIn pin cts (clear-to-send).
  */
  RWDModule(PinName tx, PinName rx, PinName cts);
  
  /*!
  Destroys instance.
  */
  //virtual ();
  
  //!Executes a command.
  /*!
  Executes the command cmd on the reader, with parameters set in params buffer of paramsLen length. The acknowledge byte sent back by the reader masked with ackOkMask  must be equal to ackOk for the command to be considered a success. If so, the result is stored in buffer resp of length respLen.
  This is a non-blocking function, and ready() should be called to check completion.
  Please note that the buffers references must remain valid until the command has been executed.
  */
  void command(uint8_t cmd, const uint8_t* params, int paramsLen, uint8_t* resp, size_t respLen, uint8_t ackOk, size_t ackOkMask); //Ack Byte is not included in the resp buf
  
  //!Ready for a command / response is available.
  /*!
  Returns true if the previous command has been executed and an other command is ready to be sent.
  */
  bool ready();
  
  //!Get whether last command was succesful, and complete ack byte if a ptr is provided.
  /*!
  Returns true if the previous command was successful. If pAck is provided, the actual acknowledge byte returned by the reader is stored in that variable.
  */
  bool result(uint8_t* pAck = NULL);

private:
  void intClearToSend(); //Called on interrupt when CTS line falls
  void intTx(); //Called on interrupt when TX buffer is not full anymore (bytes sent)
  void intRx(); //Called on interrrupt when RX buffer is not empty anymore (bytes received)

  Serial m_serial;
  InterruptIn m_cts;  
  
  uint8_t m_cmd;
  uint8_t* m_paramsBuf;
  uint8_t* m_respBuf;
  size_t m_pos;
  size_t m_paramsLen;
  size_t m_respLen;
    
  uint8_t m_ackOk;
  uint8_t m_ackOkMask;
  
  uint8_t m_ack;
  
  enum
  {
    READY,
    CMD_QUEUED,
    SENDING_CMD,
    WAITING_FOR_ACK,
    RECEIVING_ACK
  } m_state;

};

#endif