/*
Copyright (c) 2010 ARM Limited

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "RWDMifare.h"

RWDMifare::RWDMifare(PinName tx, PinName rx, PinName cts) : RWDModule(tx, rx, cts)
{

}


RWDMifare::~RWDMifare()
{

}
RWDMifare::RWDMifareErr RWDMifare::init()
{
    const uint8_t config[] = {0x03, 0x00}; //Configure ready for Mifare Op (0x00 to addr 0x03 in EEPROM)
    uint8_t resp[64] = {0};
    //Cf http://www.ibtechnology.co.uk/pdf/MF_ICprot.pdf
    //Command 'P': cf p11
    //EEPROM parameters: cf p12
    //Ack check: cf p13
    command(CMD_Prog_EEPROM, config, 2, resp, 0, 0x80, 0x89); //Program EEPROM command is 0x50 ('P')
    while(!ready()) { //Wait for a response

    }

    if(!result()) { //If this fails, there is something wrong with the hardware
        return MIFARE_HW;
    }

    return MIFARE_OK;
}
RWDMifare::RWDMifareErr RWDMifare::Prog_EEPROM(uint8_t Address, uint8_t Value)
{
    const uint8_t config[] = {Address, Value}; //Configure ready for Mifare Op (0x00 to addr 0x03 in EEPROM)
    uint8_t resp[64] = {0};
    //Cf http://www.ibtechnology.co.uk/pdf/MF_ICprot.pdf
    //Command 'P': cf p11
    //EEPROM parameters: cf p12
    //Ack check: cf p13
    command(CMD_Prog_EEPROM, config, 2, resp, 0, 0x80, 0x81); //Program EEPROM command is 0x50 ('P')
    while(!ready()) { //Wait for a response

    }

    if(!result()) { //If this fails, there is something wrong with the hardware
        return MIFARE_HW;
    }

    return MIFARE_OK;

}
RWDMifare::RWDMifareErr RWDMifare::printStatus(uint8_t Status)
{
    printf("\n\rEEPROM Error\t= %d", (Status & Ack_Code_EEPROM_ERR));
    printf("\n\rCard Okay \t= %d", (Status & Ack_Code_Card_OK)>>1);
    printf("\n\rRX Okay \t= %d", (Status & Ack_Code_RX_OK)>>2);
    printf("\n\rRS232 Error \t= %d", (Status & Ack_Code_RS232_ERR)>>3);
    printf("\n\rMF Type \t= %d", (Status & Ack_Code_MF_Type)>>4);
    printf("\n\rUL Type \t= %d", (Status & Ack_Code_UL_Type)>>5);
    printf("\n\rMFRC Error \t= %d", (Status & Ack_Code_MRRFC_Err)>>6);
    return MIFARE_OK;
}
RWDMifare::RWDMifareErr RWDMifare::getStatus(uint8_t* Status)
{
    //Gets the status code of the Mifare
    command(CMD_Get_Status, NULL, 0, Status, 1, 0x86,0x86);
    printf("\n\rStatus 0x%02x", *Status);

    return MIFARE_OK;
}
RWDMifare::RWDMifareErr RWDMifare::getUID(uint8_t* pUID, size_t* pLen) //pUID must be at least 10-bytes long
//(A Mifare UID can either be 4, 7, or 10 bytes long)
//This reader does not support 10 bytes uids
{
    //Cf http://www.ibtechnology.co.uk/pdf/MF_ICprot.pdf
    //Command 'U': cf p19
    //Ack check: cf p13
    command(CMD_Get_UID, NULL, 0, pUID, 7, 0x86, 0x86); //UID command is 0x55 ('U')
    while(!ready()) { //Wait for a response

    }

    if(!result()) { //Error detected, there is no card in field
        return MIFARE_NOCARD;
    }

    //printf("Got card.\n\r");

    //Checks UID length returned by reader
    int i;
    for(i = 0; i < 7; i++) {
        if(pUID[i] == 0) //End of UID, cf http://www.ibtechnology.co.uk/pdf/MF_ICprot.pdf p19
            break;
    }
    *pLen = i;

    return MIFARE_OK;
}

RWDMifare::RWDMifareErr RWDMifare::StoreKey(uint8_t KeyNumber, uint8_t* Key)
{
    uint8_t cmd[7] = {0};
    //command[0] = CMD_Store_Key;
    cmd[0] = KeyNumber & 31; //mask it. 
    for(int i = 0; i < 6; i++)
    {
        cmd[i+1] = Key[i];
    }
/*    printf("Command:\n");
    for(int i = 0; i < 8; i ++){
        printf("0x%02x ", command[i]);
    }*/
    command(CMD_Store_Key, cmd, 7, NULL, 0, 0x80, 0x89);
    while(!ready()) { //Wait for a response

    }

    if(!result()) { //Error detected, there is no card in field
        return MIFARE_NOCARD;
    }

    return MIFARE_OK;
}
RWDMifare::RWDMifareErr RWDMifare::AuthAllCards()
{
    if (Prog_EEPROM(0x0C, 0xFF))
        return MIFARE_HW;
    if (Prog_EEPROM(0x0C, 0xFF))
        return MIFARE_HW;
    if (Prog_EEPROM(0x0C, 0xFF))
        return MIFARE_HW;
    if (Prog_EEPROM(0x0C, 0xFF))
        return MIFARE_HW;
    return MIFARE_OK; 
}

RWDMifare::RWDMifareErr RWDMifare::ReadBlock(uint8_t Addr, uint8_t KeyNumber_Type, uint8_t* Data)
{
    uint8_t cmd[2] = {Addr, KeyNumber_Type};
    command(CMD_Read_Block, cmd, 2, Data, 16, 0x96, 0xFE);
    while(!ready()) { //Wait for a response

    }

    if(!result()) { //Error detected, there is no card in field
        return MIFARE_NOCARD;
    }

    return MIFARE_OK;
    
}
RWDMifare::RWDMifareErr RWDMifare::ReadBlock(uint8_t Addr, uint8_t KeyNumber, uint8_t Type, uint8_t* Data)
{
    KeyNumber &= 0x1F; 
    if(Type)
        KeyNumber |= 0x80;
    return ReadBlock(Addr, KeyNumber, Data);
    
}
RWDMifare::RWDMifareErr RWDMifare::WriteBlock(uint8_t Addr, uint8_t KeyNumber, uint8_t Type, uint8_t* Data)
{
    //CARE MUST BE TAKEN WHEN WRITING TO A SECTION TRAILER. 
    //THIS METHOD WILL NOT WRITE TO A SECTION TRAILER, ONLY USER DATA
    if(Addr < 32)
    {
        if ((Addr & 0x3) == 0x3) // section trailer occurs on blocks 3, 7, 11, 15 . . .
            return MIFARE_TRAIL;
    }
    else //only applicable for 4k cards 
    {
        if ((Addr & 0xF) == 0xF) //for sectors 32 - 39, trailer occurs in block 15. 
            return MIFARE_TRAIL;
    }
    
    //get here, writing to a user field.
    KeyNumber &= 0x1F; 
    if(Type)
        KeyNumber |= 0x80;
    
    uint8_t cmd[18];
    cmd[0] = Addr;
    cmd[1] = KeyNumber;
    for(int i = 0; i < 16; i ++)
    {
        cmd[i+2] = Data[i];
    }
    printf("Command to write:\n");
    for(int i = 0; i < 18;i++){
        printf("0x%02x ", cmd[i]);
    }
    printf("\r\n");
    command(CMD_Write_Block, cmd, 18, NULL, 0, 0x96, 0xFE);
    while(!ready()) { //Wait for a response

    }

    if(!result()) { //Error detected, there is no card in field
        return MIFARE_NOCARD;
    }
    return MIFARE_OK;
}