/*!
 ******************************************************************************
 * @file:   adi_sense_log.h
 * @brief:  ADI Sense OS-dependent wrapper layer for log functions
 *-----------------------------------------------------------------------------
 */

/*
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef __ADI_SENSE_LOG_H__
#define __ADI_SENSE_LOG_H__

#include "inc/adi_sense_types.h"

/*! @ingroup ADI_Sense_Host */

/*! @addtogroup ADI_Sense_Log ADI Sense Host Logging functions
 *  @{
 */

/*! Macro function for logging an error message */
#define ADI_SENSE_LOG_ERROR(...)                                        \
    adi_sense_Log(ADI_SENSE_LOG_LEVEL_ERROR, "[ERROR] " __VA_ARGS__)
/*! Macro function for logging a warning message */
#define ADI_SENSE_LOG_WARN(...)  \
    adi_sense_Log(ADI_SENSE_LOG_LEVEL_WARN,  "[WARN] "  __VA_ARGS__)
/*! Macro function for logging an information message */
#define ADI_SENSE_LOG_INFO(...)  \
    adi_sense_Log(ADI_SENSE_LOG_LEVEL_INFO,  "[INFO] "  __VA_ARGS__)
/*! Macro function for logging a debug message */
#define ADI_SENSE_LOG_DEBUG(...) \
    adi_sense_Log(ADI_SENSE_LOG_LEVEL_DEBUG, "[DEBUG] " __VA_ARGS__)

/*!
 * Log message priority levels
 */
typedef enum
{
    ADI_SENSE_LOG_LEVEL_ERROR = 0, /*!< Error message priority */
    ADI_SENSE_LOG_LEVEL_WARN,      /*!< Warning message priority */
    ADI_SENSE_LOG_LEVEL_INFO,      /*!< Information message priority */
    ADI_SENSE_LOG_LEVEL_DEBUG,     /*!< Debug message priority */
} ADI_SENSE_LOG_LEVEL;


#ifdef __cplusplus
extern "C" 
{
#endif

/*!
 * @brief Initialise the Log interface and allocate resources.
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *         - #ADI_SENSE_NO_MEM  Failed to allocate memory for device context.
 */
ADI_SENSE_RESULT adi_sense_LogOpen(
    void);

/*!
 * @brief Close the Log interface and free resources.
 */
void adi_sense_LogClose(
    void);

/*!
 * @brief Print a log message to the platform log interface.
 *
 * @param[in] level  Log message priority level
 * @param[in] format Format string and variable argument list, if any
 */
void adi_sense_Log(
    ADI_SENSE_LOG_LEVEL   level,
    const char          * format,
    ...);

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_LOG_H__ */

