/*!
 ******************************************************************************
 * @file:   adi_sense_gpio.cpp
 * @brief:  ADI Sense OS-dependent wrapper layer for GPIO interface
 *-----------------------------------------------------------------------------
 */

/******************************************************************************
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

#include <mbed.h>

#include "inc/adi_sense_gpio.h"
#include "inc/adi_sense_log.h"

class GpioContext
{
public:
    GpioContext(
        PinName resetPin,
        PinName errorPin,
        PinName alertPin,
        PinName datareadyPin)
        : _reset(resetPin),
          _error(errorPin),
          _alert(alertPin),
          _dataready(datareadyPin),
          _errorIrq(errorPin),
          _alertIrq(alertPin),
          _datareadyIrq(datareadyPin) {}

    ADI_SENSE_RESULT get(
        ADI_SENSE_GPIO_PIN ePinId,
        bool_t *pState);

    ADI_SENSE_RESULT set(
        ADI_SENSE_GPIO_PIN ePinId,
        bool_t state);

    ADI_SENSE_RESULT enableIrq(
        ADI_SENSE_GPIO_PIN ePinId,
        ADI_SENSE_GPIO_CALLBACK callbackFn,
        void *pArg);

    ADI_SENSE_RESULT disableIrq(
        ADI_SENSE_GPIO_PIN ePinId);

private:
    DigitalOut  _reset;

    DigitalIn   _error;
    DigitalIn   _alert;
    DigitalIn   _dataready;

    InterruptIn _errorIrq;
    InterruptIn _alertIrq;
    InterruptIn _datareadyIrq;

    ADI_SENSE_GPIO_CALLBACK _errorIrqCallback;
    ADI_SENSE_GPIO_CALLBACK _alertIrqCallback;
    ADI_SENSE_GPIO_CALLBACK _datareadyIrqCallback;

    void *_errorIrqArg;
    void *_alertIrqArg;
    void *_datareadyIrqArg;

    void _errorIrqHandler()
    {
        _errorIrqCallback(ADI_SENSE_GPIO_PIN_ERROR, _errorIrqArg);
    }
    void _alertIrqHandler()
    {
        _alertIrqCallback(ADI_SENSE_GPIO_PIN_ALERT, _alertIrqArg);
    }
    void _datareadyIrqHandler()
    {
        _datareadyIrqCallback(ADI_SENSE_GPIO_PIN_DATAREADY, _datareadyIrqArg);
    }
};

ADI_SENSE_RESULT GpioContext::get(
    ADI_SENSE_GPIO_PIN ePinId,
    bool_t *pState)
{
    switch(ePinId)
    {
    case ADI_SENSE_GPIO_PIN_ERROR:
        *pState = _error;
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_ALERT:
        *pState = _alert;
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_DATAREADY:
        *pState = _dataready;
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_RESET:
        *pState = _reset;
        return ADI_SENSE_SUCCESS;
    default:
        return ADI_SENSE_INVALID_DEVICE_NUM;
    }
}

ADI_SENSE_RESULT GpioContext::set(
    ADI_SENSE_GPIO_PIN ePinId,
    bool_t state)
{
    switch(ePinId)
    {
    case ADI_SENSE_GPIO_PIN_RESET:
        _reset = state;
        break;
    default:
        return ADI_SENSE_INVALID_DEVICE_NUM;
    }

    return ADI_SENSE_SUCCESS;
}

ADI_SENSE_RESULT GpioContext::enableIrq(
    ADI_SENSE_GPIO_PIN ePinId,
    ADI_SENSE_GPIO_CALLBACK callbackFn,
    void *pArg)
{
    switch(ePinId)
    {
    case ADI_SENSE_GPIO_PIN_ERROR:
        _errorIrqCallback = callbackFn;
        _errorIrqArg = pArg;
        _errorIrq.rise(callback(this, &GpioContext::_errorIrqHandler));
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_ALERT:
        _alertIrqCallback = callbackFn;
        _alertIrqArg = pArg;
        _alertIrq.rise(callback(this, &GpioContext::_alertIrqHandler));
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_DATAREADY:
        _datareadyIrqCallback = callbackFn;
        _datareadyIrqArg = pArg;
        _datareadyIrq.rise(callback(this, &GpioContext::_datareadyIrqHandler));
        return ADI_SENSE_SUCCESS;
    default:
        return ADI_SENSE_INVALID_DEVICE_NUM;
    }
}

ADI_SENSE_RESULT GpioContext::disableIrq(
    ADI_SENSE_GPIO_PIN ePinId)
{
    switch(ePinId)
    {
    case ADI_SENSE_GPIO_PIN_ERROR:
        _errorIrq.rise(NULL);
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_ALERT:
        _alertIrq.rise(NULL);
        return ADI_SENSE_SUCCESS;
    case ADI_SENSE_GPIO_PIN_DATAREADY:
        _datareadyIrq.rise(NULL);
        return ADI_SENSE_SUCCESS;
    default:
        return ADI_SENSE_INVALID_DEVICE_NUM;
    }
}

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Open the GPIO interface and allocate resources
 */
ADI_SENSE_RESULT adi_sense_GpioOpen(
    ADI_SENSE_PLATFORM_GPIO_CONFIG *pConfig,
    ADI_SENSE_GPIO_HANDLE *phDevice)
{
    GpioContext *pCtx = new GpioContext((PinName)pConfig->resetPin,
                                        (PinName)pConfig->errorPin,
                                        (PinName)pConfig->alertPin,
                                        (PinName)pConfig->datareadyPin);
    if (!pCtx)
    {
        ADI_SENSE_LOG_ERROR("Failed to allocate memory for GPIO context");
        return ADI_SENSE_NO_MEM;
    }

    *phDevice = reinterpret_cast<ADI_SENSE_GPIO_HANDLE>(pCtx);
    return ADI_SENSE_SUCCESS;
}

/*
 * Get the state of the specified GPIO pin
 */
ADI_SENSE_RESULT adi_sense_GpioGet(
    ADI_SENSE_GPIO_HANDLE hDevice,
    ADI_SENSE_GPIO_PIN ePinId,
    bool_t *pbState)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->get(ePinId, pbState);
}

/*
 * Set the state of the specified GPIO pin
 */
ADI_SENSE_RESULT adi_sense_GpioSet(
    ADI_SENSE_GPIO_HANDLE hDevice,
    ADI_SENSE_GPIO_PIN ePinId,
    bool_t bState)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->set(ePinId, bState);
}

/*
 * Enable interrupt notifications on the specified GPIO pin
 */
ADI_SENSE_RESULT adi_sense_GpioIrqEnable(
    ADI_SENSE_GPIO_HANDLE hDevice,
    ADI_SENSE_GPIO_PIN ePinId,
    ADI_SENSE_GPIO_CALLBACK callback,
    void *arg)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->enableIrq(ePinId, callback, arg);
}

/*
 * Disable interrupt notifications on the specified GPIO pin
 */
ADI_SENSE_RESULT adi_sense_GpioIrqDisable(
    ADI_SENSE_GPIO_HANDLE hDevice,
    ADI_SENSE_GPIO_PIN ePinId)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->disableIrq(ePinId);
}

/*
 * Close the GPIO interface and free resources
 */
void adi_sense_GpioClose(
    ADI_SENSE_GPIO_HANDLE hDevice)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    delete pCtx;
}

#ifdef __cplusplus
}
#endif

