#include "mbed.h"
#include <iostream>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <locale.h> 
#include "rtos.h"
#include "SLCD.h"

#define  PRINTDBUG
#define PROGNAME    "DAC OUT serial KL46Z v3\n\r"
//Code parameters

#define VOLTINC    0.04f // 25 points
//LCD messages
#define VOLTAST      0
#define DATARATE    0.5 // sec
#define MAXVOLTS    3.3
#define MAXPERCENT  100.00f
#define FUTURESTATE 1
#define LCDLEN      20
#define MAXCHAR     5
#define NUMLEDS     3
#define LEDOFF      1
#define LEDON       0
#define GREENL      1
#define REDL        0
#define BLUEL       2

#define SERIALREADTIME  50 //ms
#define LEDBLINKTIME    300 // ms *** NOTE Change of units ***
#define LCDUPDATE       100 //ms
// ASCII constants
#define CR              '\r'
#define ALLNUMS        "1234567890. " // 10 numbers + decimal pt
#define NOTDATA         12
#define VOLTSTOKEN      'V'
#define QTOKEN          'Q'
#define WHOTOKEN        'W'
#define HLPTOKEN        '?'
#define CURSRTOKEN      ':'

#define COMMANDLEN      7 //Max characters in the command
#define NOTOK           "NOK"
#define NOTVALID        2
#define VALID           1
#define VINDEX          0
#define NUMINDEX        1


// Help Messages
#define HLEN            35
#define NUMHLINES       5       
#define HLINE1          "\r\n - Usage -"
#define HLINE2          "Vxx.xx - setpoint as % full scale,"
#define HLINE3          "Q - Show current setpoint."
#define HLINE4          "W - Show version (Who am I)."
#define HLINE5          "? - Usage Help - (This message)."

char displayHelpLines[NUMHLINES][HLEN] = {HLINE1,HLINE2,HLINE3,HLINE4,HLINE5};

//Global classes
AnalogOut refOut(PTE30);

DigitalOut allLEDS[NUMLEDS]= {LED_RED,LED_GREEN,LED_RED};
 
Serial pc(USBTX, USBRX); // tx, rx

char rxChar;
char rxString[LCDLEN];
char numString[LCDLEN];
char LCDActual[LCDLEN];
SLCD slcd; //define LCD display
bool commandReady = false; // This is the flag to even look at the command string
float floatVolts = 0.0f; 
int ledErrorCode = GREENL;


void LCDMessNoDwell(char *lMess){
        slcd.Home();
        slcd.clear();
        slcd.printf(lMess);
}

// use "thread" in the name to keep things straight 
// note the use of void constant * args - understand memory resources
// Thes are "forever loops"
void LCDdis_thread(void const *args){
    while(true) {
        LCDMessNoDwell(LCDActual);
        Thread::wait(LCDUPDATE);
    }
}

void clearString (char * aString){
    int i;
    int sSize = strlen(aString);
    
    for (i=0; i< sSize; i++){
        aString[i] = NULL;
    }
    return;
}

void showCursor(){
    pc.printf("%c", CURSRTOKEN);
    return;
}

void showProgName(bool singleChar){
    pc.printf("\n\r");
    switch (singleChar){
        case true: {    
            pc.printf(PROGNAME);
            ledErrorCode = GREENL;
            break;
        }
        case false: {
            pc.printf("%s - NOK\n\r", rxString);
            ledErrorCode = BLUEL; 
            break;
        }
    }   
    clearString(rxString);
    clearString(numString);
    showCursor();
    return;
}    

void showHelpLines(bool singleChar){
    int i;
    switch (singleChar){
        case true: {   
            for (i=0; i< NUMHLINES; i++){
            pc.printf("%s\n\r", displayHelpLines[i]);
            }
            ledErrorCode = GREENL;
            break;
        }
        case false: {
            pc.printf("%s - NOK\n\r", rxString);
            ledErrorCode = BLUEL; 
            break;
        }
     }
    clearString(rxString);
    clearString(numString);
    showCursor();
    return;
}


void allLEDSOff() {
    int i;
    for (i=0; i<NUMLEDS; i++) {
        allLEDS[i].write(LEDOFF);
    }
    return;
}
void oneLEDSet(int oneLedIndex, int oneLedState){
    allLEDSOff();
    allLEDS[oneLedIndex].write(oneLedState);
    return;
}

int isValidData (char dataChar , char * validDataChars){
    int noteListLen = strlen(validDataChars);
    int pos = noteListLen;
    int i;
    
    for (i=0; i<noteListLen; i++){
        if (validDataChars[i] == dataChar){
            pos = i;
            break;
        }
    }
    return (pos);
}
// calculating the valid command
void parseVCommand (char * commandChars){
    int charIndex;
    int comLen;
    bool validCommand = true;
    int isNum;
    int parseState = NOTVALID;
    static float lastValue = 0.0f;
    
    pc.printf("\n\r");
    comLen = strlen(commandChars);
    // validCommand = (commandChars[VINDEX] == VOLTSTOKEN);
    for (charIndex = NUMINDEX; charIndex < comLen; charIndex++){
        if (!validCommand) {
            parseState = NOTVALID;
            break;
        }
        isNum = isValidData(commandChars[charIndex], ALLNUMS);
        numString[charIndex - NUMINDEX] = commandChars[charIndex]; // make number
        validCommand = (isNum != NOTDATA) && validCommand;
        parseState = VALID;        
    }
    
    switch (parseState) {
        case NOTVALID: {
            pc.printf("%s - NOK\n\r", rxString);
            ledErrorCode = BLUEL; 
            break;
            }
            default: {
                floatVolts = atof(numString);
                if (floatVolts >= MAXPERCENT){
                    pc.printf("%5.2f Value too large NOK\n\r", floatVolts); 
                    floatVolts = lastValue;
                   ledErrorCode = REDL; 
                }else {
                    pc.printf("%5.2f Set Voltage\n\r", floatVolts); 
                    lastValue = floatVolts;
                    ledErrorCode = GREENL;
                }
                break;
            }
        }
    clearString(rxString);
    clearString(numString);
    showCursor();
    return;
}
void parseQCommand (char * commandChars){
    int parseState = NOTVALID;
    
    pc.printf("\n\r");
    parseState = strlen(commandChars);
    
    switch (parseState) {
        case VALID : {
            pc.printf("%5.2f Q-Set Voltage\n\r", floatVolts);
            ledErrorCode = GREENL; 
            break; 
        }          
        default: {
            pc.printf("%s - Q - NOK\n\r", rxString);
            ledErrorCode = BLUEL; 
            break;
            }
    }
    clearString(rxString);
    clearString(numString);
    showCursor();
    return;
}
// use "thread" in the name to keep things straight 
// note the use of void constant * args - understand memory resources
// Thes are "forever loops"



void serial_thread(void const *args){
   static int charIndex =0;
   char firstChar;
   bool charOK;
   
   while(true) {
    if(charIndex >= COMMANDLEN){
        charIndex = 0;
        clearString(rxString);
        clearString(numString);
        pc.printf("  ComLen exceeded - NOK\n\r");  
        ledErrorCode = BLUEL; 
        showCursor();                
    }    
      if (pc.readable()) {                // only read from the serial port if there is a character
            rxChar= toupper(pc.getc());           // reading clears the buffer      
            // check for carriage return 
            switch (rxChar){
                case CR: { 
                    firstChar =  rxString[VINDEX];
                    charOK = (strlen(rxString) == 1);
                    switch (firstChar){
                        case VOLTSTOKEN :{
                            parseVCommand(rxString);     // command length it's not a good command                           
                            break;
                        }
                        case QTOKEN: {
                            parseQCommand(rxString);
                            break;
                        }
                        case  WHOTOKEN: {
                            showProgName(charOK);
                            break;
                        }             
                        case HLPTOKEN: {
                            showHelpLines(charOK);
                            break;
                        }                      
                        default: {
                            pc.printf("\n\r%s - Comm NOK\n\r", rxString);
                            ledErrorCode = BLUEL; 
                            showCursor();
                            break;
                        }
                    }
                    charIndex = 0; // start building string from position zero next time around   
                    break;
                }
                default: {      
                    rxString[charIndex] = rxChar;
                    pc.printf("%c", rxChar);  // Echo input characters
                    charIndex++; 
                    break;
                }  
            }
      }
      Thread::wait(SERIALREADTIME);
    }
}

void parameterAdjust( int adjState, float scaling) { // small adjustment state machine
    float  tempFloat;
    switch (adjState){
        case VOLTAST: {
             tempFloat = scaling;
             refOut.write(tempFloat);              
             break;
        }
        case FUTURESTATE: {
        
             break;
        }
    }
    return;
}

int main(){
    
    float tempValue=0.0f;
    int ledState = true;
   
    Thread serthread(serial_thread);
    Thread lthread(LCDdis_thread);
    
    allLEDSOff();
    showProgName(true);
    showHelpLines(true);
    parameterAdjust( VOLTAST, 0.0);
    
    sprintf (LCDActual,"%4.2f",floatVolts);
    while (true) {
        tempValue = floatVolts / MAXPERCENT; // do this for later adjustment.
        if(tempValue >= 0) {
                    parameterAdjust( VOLTAST, tempValue);
                    sprintf (LCDActual,"%4.2f",floatVolts); // For LCD Display
                    oneLEDSet(ledErrorCode, ledState);
        }       
       ledState = !ledState;
       Thread::wait(LEDBLINKTIME);
    } // while forever
}// end main