#include <mbed.h>

/*=========================================================================
    I2C ADDRESS/BITS
    -----------------------------------------------------------------------*/
    #define ADS1115_ADDRESS                 (0x49)    // 100 1001 (ADDR = VDD)
/*=========================================================================*/

/*=========================================================================
    CONVERSION DELAY (in mS)
    -----------------------------------------------------------------------*/
    #define ADS1115_CONVERSIONDELAY         (8)
/*=========================================================================*/

/*=========================================================================
    POINTER REGISTER
    -----------------------------------------------------------------------*/
    #define ADS1115_REG_POINTER_MASK        (0x03)
    #define ADS1115_REG_POINTER_CONVERT     (0x00)
    #define ADS1115_REG_POINTER_CONFIG      (0x01)
    #define ADS1115_REG_POINTER_LOWTHRESH   (0x02)
    #define ADS1115_REG_POINTER_HITHRESH    (0x03)
/*=========================================================================*/

/*=========================================================================
    CONFIG REGISTER
    -----------------------------------------------------------------------*/
    #define ADS1115_REG_CONFIG_OS_MASK      (0x8000)
    #define ADS1115_REG_CONFIG_OS_SINGLE    (0x8000)  // Write: Set to start a single-conversion
    #define ADS1115_REG_CONFIG_OS_BUSY      (0x0000)  // Read: Bit = 0 when conversion is in progress
    #define ADS1115_REG_CONFIG_OS_NOTBUSY   (0x8000)  // Read: Bit = 1 when device is not performing a conversion

    #define ADS1115_REG_CONFIG_MUX_MASK     (0x7000)
    #define ADS1115_REG_CONFIG_MUX_DIFF_0_1 (0x0000)  // Differential P = AIN0, N = AIN1 (default)
    #define ADS1115_REG_CONFIG_MUX_DIFF_0_3 (0x1000)  // Differential P = AIN0, N = AIN3
    #define ADS1115_REG_CONFIG_MUX_DIFF_1_3 (0x2000)  // Differential P = AIN1, N = AIN3
    #define ADS1115_REG_CONFIG_MUX_DIFF_2_3 (0x3000)  // Differential P = AIN2, N = AIN3
    #define ADS1115_REG_CONFIG_MUX_SINGLE_0 (0x4000)  // Single-ended AIN0
    #define ADS1115_REG_CONFIG_MUX_SINGLE_1 (0x5000)  // Single-ended AIN1
    #define ADS1115_REG_CONFIG_MUX_SINGLE_2 (0x6000)  // Single-ended AIN2
    #define ADS1115_REG_CONFIG_MUX_SINGLE_3 (0x7000)  // Single-ended AIN3

    #define ADS1115_REG_CONFIG_PGA_MASK     (0x0E00)
    #define ADS1115_REG_CONFIG_PGA_6_144V   (0x0000)  // +/-6.144V range = Gain 2/3
    #define ADS1115_REG_CONFIG_PGA_4_096V   (0x0200)  // +/-4.096V range = Gain 1
    #define ADS1115_REG_CONFIG_PGA_2_048V   (0x0400)  // +/-2.048V range = Gain 2 (default)
    #define ADS1115_REG_CONFIG_PGA_1_024V   (0x0600)  // +/-1.024V range = Gain 4
    #define ADS1115_REG_CONFIG_PGA_0_512V   (0x0800)  // +/-0.512V range = Gain 8
    #define ADS1115_REG_CONFIG_PGA_0_256V   (0x0A00)  // +/-0.256V range = Gain 16

    #define ADS1115_REG_CONFIG_MODE_MASK    (0x0100)
    #define ADS1115_REG_CONFIG_MODE_CONTIN  (0x0000)  // Continuous conversion mode
    #define ADS1115_REG_CONFIG_MODE_SINGLE  (0x0100)  // Power-down single-shot mode (default)

    #define ADS1115_REG_CONFIG_DR_MASK      (0x00E0)  
    #define ADS1115_REG_CONFIG_DR_128SPS      (0x0000)  // 128SPS
    #define ADS1115_REG_CONFIG_DR_250SPS     (0x0020)  // 250SPS
    #define ADS1115_REG_CONFIG_DR_490SPS     (0x0040)  // 490SPS
    #define ADS1115_REG_CONFIG_DR_920SPS     (0x0060)  // 920SPS
    #define ADS1115_REG_CONFIG_DR_1600SPS    (0x0080)  // 1600SPS (default)
    #define ADS1115_REG_CONFIG_DR_2400SPS    (0x00A0)  // 2400SPS
    #define ADS1115_REG_CONFIG_DR_3300SPS    (0x00C0)  // 3300SPS
    #define ADS1115_REG_CONFIG_DR_3300SPS    (0x00E0)  // 3300SPS
    
//    #define ADS1115_REG_CONFIG_DR_8SPS      (0x0000)  // 8SPS
//    #define ADS1115_REG_CONFIG_DR_16SPS     (0x0020)  // 16SPS
//    #define ADS1115_REG_CONFIG_DR_32SPS     (0x0040)  // 32SPS
//    #define ADS1115_REG_CONFIG_DR_64SPS     (0x0060)  // 64SPS
//    #define ADS1115_REG_CONFIG_DR_128SPS    (0x0080)  // 128SPS (default)
//    #define ADS1115_REG_CONFIG_DR_475SPS    (0x00A0)  // 475SPS
//    #define ADS1115_REG_CONFIG_DR_860SPS    (0x00C0)  // 860SPS

    #define ADS1115_REG_CONFIG_CMODE_MASK   (0x0010)
    #define ADS1115_REG_CONFIG_CMODE_TRAD   (0x0000)  // Traditional comparator with hysteresis (default)
    #define ADS1115_REG_CONFIG_CMODE_WINDOW (0x0010)  // Window comparator

    #define ADS1115_REG_CONFIG_CPOL_MASK    (0x0008)
    #define ADS1115_REG_CONFIG_CPOL_ACTVLOW (0x0000)  // ALERT/RDY pin is low when active (default)
    #define ADS1115_REG_CONFIG_CPOL_ACTVHI  (0x0008)  // ALERT/RDY pin is high when active

    #define ADS1115_REG_CONFIG_CLAT_MASK    (0x0004)  // Determines if ALERT/RDY pin latches once asserted
    #define ADS1115_REG_CONFIG_CLAT_NONLAT  (0x0000)  // Non-latching comparator (default)
    #define ADS1115_REG_CONFIG_CLAT_LATCH   (0x0004)  // Latching comparator

    #define ADS1115_REG_CONFIG_CQUE_MASK    (0x0003)
    #define ADS1115_REG_CONFIG_CQUE_1CONV   (0x0000)  // Assert ALERT/RDY after one conversions
    #define ADS1115_REG_CONFIG_CQUE_2CONV   (0x0001)  // Assert ALERT/RDY after two conversions
    #define ADS1115_REG_CONFIG_CQUE_4CONV   (0x0002)  // Assert ALERT/RDY after four conversions
    #define ADS1115_REG_CONFIG_CQUE_NONE    (0x0003)  // Disable the comparator and put ALERT/RDY in high state (default)
/*=========================================================================*/

typedef enum {
    GAIN_TWOTHIRDS    = ADS1115_REG_CONFIG_PGA_6_144V,
    GAIN_ONE          = ADS1115_REG_CONFIG_PGA_4_096V,
    GAIN_TWO          = ADS1115_REG_CONFIG_PGA_2_048V,
    GAIN_FOUR         = ADS1115_REG_CONFIG_PGA_1_024V,
    GAIN_EIGHT        = ADS1115_REG_CONFIG_PGA_0_512V,
    GAIN_SIXTEEN      = ADS1115_REG_CONFIG_PGA_0_256V
} adsGain_t;

uint8_t   m_i2cAddress;
uint8_t   m_conversionDelay;
uint8_t   m_bitShift;
adsGain_t m_gain;
I2C*      m_i2c;

void ADS1115_Init(I2C* i2c, uint8_t i2cAddress)
{
    // shift 7 bit address 1 left: read expects 8 bit address, see I2C.h
    m_i2cAddress = i2cAddress << 1;
    m_conversionDelay = ADS1115_CONVERSIONDELAY;
    m_gain = GAIN_TWOTHIRDS; /* +/- 6.144V range (limited to VDD +0.3V max!) */
    m_i2c = i2c;
}

uint16_t ADS1115_readRegister(uint8_t i2cAddress, uint8_t reg)
{
    char data[2];
    data[0] = reg; // temporary use this to send address to conversion register
    m_i2c->write(i2cAddress, data, 1);
    m_i2c->read(i2cAddress, data, 2);
    return (data[0] << 8 | data [1]);
}

void ADS1115_writeRegister(uint8_t i2cAddress, uint8_t reg, uint16_t value)
{
    char cmd[3];
    cmd[0] = (char)reg;
    cmd[1] = (char)(value>>8);
    cmd[2] = (char)(value & 0xFF);
    m_i2c->write(i2cAddress, cmd, 3);
}


uint16_t ADS1115_setConti_ADC_SingleEnded(uint8_t channel)
{
    if (channel > 3) {
        return 0;
    }

    // Start with default values
    uint16_t config = ADS1115_REG_CONFIG_CQUE_1CONV   | // Assert ALERT/RDY after one conversions
                      ADS1115_REG_CONFIG_CLAT_NONLAT  | // Non-latching (default val)
                      ADS1115_REG_CONFIG_CPOL_ACTVLOW | // Alert/Rdy active low   (default val)
                      ADS1115_REG_CONFIG_CMODE_TRAD   | // Traditional comparator (default val)
                      ADS1115_REG_CONFIG_DR_490SPS    | // 490SPS
                      ADS1115_REG_CONFIG_MODE_CONTIN;   // Countinuous-conversion mode 

    // Set PGA/voltage range
    config |= m_gain;

    // Set single-ended input channel
    switch (channel) {
        case (0):
            config |= ADS1115_REG_CONFIG_MUX_SINGLE_0;
            break;
        case (1):
            config |= ADS1115_REG_CONFIG_MUX_SINGLE_1;
            break;
        case (2):
            config |= ADS1115_REG_CONFIG_MUX_SINGLE_2;
            break;
        case (3):
            config |= ADS1115_REG_CONFIG_MUX_SINGLE_3;
            break;
    }   
    ADS1115_writeRegister(m_i2cAddress, ADS1115_REG_POINTER_LOWTHRESH, 0x7fff);
    ADS1115_writeRegister(m_i2cAddress, ADS1115_REG_POINTER_HITHRESH, 0x8000);
    ADS1115_writeRegister(m_i2cAddress, ADS1115_REG_POINTER_CONFIG, config);
    
    return 0;
}

int16_t ADS1115_getConti_ADC_SingleEnded()
{
    uint16_t res = ADS1115_readRegister(m_i2cAddress, ADS1115_REG_POINTER_CONVERT);
    return (int16_t)res;
}