/**
 * Rapid-prototyping protection schemes with IEC 61850
 *
 * Copyright (c) 2011 Steven Blair
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "mbed.h"
#include "iec61850.h"

#define TWO_PI              6.283185307179586476925286766559
#define TWO_PI_OVER_THREE   2.0943951023931954923084289221863
#define MAGNITUDE           100
#define INCREMENT           0.39269908169872415480783042290994  // 2 * pi / 16
#define ERROR               (1.0 + (rand() % 5) / 100.0)

unsigned char bufOut[2048] = {0};
unsigned char bufIn[2048] = {0};
int waitTime = 0;
int lenIn = 0;
int lenOut = 0;
float phase = 0.0;
float MAGNITUDE_NEG_SEQ = 0.0;
float MAGNITUDE_ZERO_SEQ = 0.0;
float offset = 0.0;

void setFault() {
    MAGNITUDE_NEG_SEQ = 50.0;
    MAGNITUDE_ZERO_SEQ = 25.0;
}

void setNormal() {
    MAGNITUDE_NEG_SEQ = 0.0;
    MAGNITUDE_ZERO_SEQ = 0.0;
    offset = 0.0;
}

DigitalOut watchdogLED(LED1);
DigitalOut inputLED(LED4);
Ethernet eth;
DigitalOut ethLink(p29);
DigitalOut ethAct(p30);
Ticker sv;

void svSnapshot() {
    E1Q1SB1.S1.C1.RMXU_1.AmpLocPhsA.instMag.f = MAGNITUDE * sin(phase + offset) * ERROR + MAGNITUDE_NEG_SEQ * sin(phase + offset) + MAGNITUDE_ZERO_SEQ * sin(phase + offset);
    E1Q1SB1.S1.C1.RMXU_1.AmpLocPhsB.instMag.f = MAGNITUDE * sin(phase + offset - TWO_PI_OVER_THREE) * ERROR + MAGNITUDE_NEG_SEQ * sin(phase + offset + TWO_PI_OVER_THREE) + MAGNITUDE_ZERO_SEQ * sin(phase + offset);
    E1Q1SB1.S1.C1.RMXU_1.AmpLocPhsC.instMag.f = MAGNITUDE * sin(phase + offset + TWO_PI_OVER_THREE) * ERROR + MAGNITUDE_NEG_SEQ * sin(phase + offset - TWO_PI_OVER_THREE) + MAGNITUDE_ZERO_SEQ * sin(phase + offset);

    lenOut = sv_update_rmxuCB_rmxu(bufOut);
    
    if (lenOut > 0) {
        ethAct = 1;
        eth.write((const char *) bufOut, lenOut);
        eth.send();
        ethAct = 0;
    }
    
    phase += INCREMENT;
    if (phase > TWO_PI) {
        phase = phase - TWO_PI;
    }
}

int main() {
    initialise_iec61850();
    
    eth.set_link(eth.FullDuplex100);
    while (!eth.link() && waitTime++ < 60) {
        wait(1);
    }
    ethLink = 1;
    
    wait(1);
    
    sv.attach_us(&svSnapshot, 1250);

    wait(5);
    
    while(1) {
        wait_us(100);
        
        lenIn = eth.receive();
        if (lenIn == 272) {
            eth.read((char *) bufIn, lenIn);
            gseDecode(bufIn, lenIn);
            
            //printf("%f\n", D1Q1SB4.S1.C1.RSYN_1.gse_inputs.instMag_1.f);  // monitor values from serial port
            
            if (D1Q1SB4.S1.C1.RSYN_1.gse_inputs.instMag_1.f > 0) {
                setFault();
            }
            else {
                setNormal();
            }
        }
    }
}
