/**
 * mbed library for 4D Systems uOLED-160-G1
 *
 *
 * Copyright (c) 2010 Steven Blair
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef _OLED160G1_H_
#define _OLED160G1_H_


#define OLED_BAUDRATE                       115200  // 300 baud to 256kbaud supported (256k not always stable?)
#define OLED_INIT_DELAY                     1100    // milliseconds

// Initialisation routine
#define OLED_DETECT_BAUDRATE                0x55

// Drawing routines
#define OLED_CLEAR                          0x45
#define OLED_BKGCOLOR                       0x42
#define OLED_COPYPASTE                      0x63
#define OLED_PEN_SIZE                       0x70

// Graphics
#define OLED_LINE                           0x4C
#define OLED_CIRCLE                         0x43
#define OLED_CIRCLEFILL                     0x69
#define OLED_PUTPIXEL                       0x50
#define OLED_READPIXEL                      0x52
#define OLED_RECTANGLE                      0x72

// Text properties
#define OLED_TEXT                           0x73
#define OLED_SETFONTSIZE                    0x46
#define OLED_FONT5X7                        0x01
#define OLED_FONT8X8                        0x02
#define OLED_FONT8X12                       0x03
#define OLED_TEXTFORMATED                   0x54
#define OLED_SET_TEXT_BACKGROUND_TYPE       0x4F
#define OLED_SET_BACKGROUND_COLOR           0x42
#define OLED_SET_TEXT_TRANSPARENT           0x00
#define OLED_SET_TEXT_OPAQUE                0x01

// OLED Control
#define OLED_COMMAND_CONTROL                0x59
#define OLED_COMMAND_DISPLAY                0x01
#define OLED_COMMAND_CONTRAST               0x02
#define OLED_COMMAND_POWER                  0x03

// Responses
#define OLED_ACK                            0x06  // Ok
#define OLED_NAK                            0x15  // Error

// Colours
#define OLED_WHITE                          0xFFFF
#define OLED_BLACK                          0x0000
#define OLED_BLUE                           0xA6BF
#define OLED_RED                            0xF800

#define red_min 0//150
#define red_max 255
#define blue_min 0//185
#define blue_max 255
#define green_min 0//195
#define green_max 255


class OLED160G1 : public Stream {
public:

    /**
     * Default constructor.
     */
    OLED160G1(PinName serialTx, PinName serialRx, PinName resetPin);
    
    /**
     * Reset the display using the reset pin (the reset pin is active-low).
     */
    void resetDisplay();

    /**
     * Initialise OLED display.
     */
    void init();

    /**
     * Processes responses (ACK or NAK) from the OLED. A new command cannot be sent to the OLED until a NAK is received, so
     * this function waits for the minimum time needed.
     */
    void getResponse();

    /**
     * Clear the OLED screen.
     */
    void eraseScreen();
    
    /**
     * Calculate 16-bit value from RGB (0 to 63, 565 format)
     */
    int toRGB(int red, int green, int blue);
    
    /**
     * 
     */
    void drawPixel(char x, char y, int color);
    
    /**
     * 
     */
    void drawLine(char x1, char y1, char x2, char y2, int color);
    
    /**
     * 
     */
    void drawRectangle(char x, char y, char width, char height, int color);
    
    /**
     * 
     */
    void drawCircle(char x, char y, char radius, int color);
    
    /**
     * Set font size, for use with printf() or OLED_DrawSingleChar(); all other functions override this setting.
     *
     * @param : fontSize can be: OLED_FONT5X7, OLED_FONT8X8, or OLED_FONT8X12
     */
    void setFontSize(char fontSize);
    
    /**
     * Set font color, for use with printf(); all other functions override this setting.
     */
    void setFontColor(int fontColor);
    
    /**
     * Set the "pen size".
     *
     * @param penSize : 0 to draw solid objects; 1 to draw wireframe objects.
     */
    void setPenSize(char penSize);
    
    /**
     * Set whether or not text is drawn with an opaque or transparent background.
     *
     * @param : textBackgroundType can be OLED_SET_TEXT_TRANSPARENT or OLED_SET_TEXT_OPAQUE
     */
    void setTextBackgroundType(char textBackgroundType);
    
    /**
     * 
     */
    void setBackgroundColor(int color);
    
    /**
     * 
     */
    void drawText(char column, char row, char font_size, char *mytext, int color);
    
    /**
     * Draw a single ASCII character at the specified location.
     */
    void drawSingleChar(char column, char row, char theChar, int color);
    
    /**
     * Display control functions, such as display ON/OFF, contrast and power-up/power-down.
     *
     * @param value : can be 0x00 to 0x0F is mode is OLED_COMMAND_CONTRAST. Default contrast value is 0x08.
     */
    void displayControl(char mode, char value);
    
    /**
     * 
     */
    char getPenSize();
    
    /**
     * Get number of text rows
     */
    int rows(); //TODO: must depend on font size
    
    /**
     * Get number of text columns
     */
    int columns();
    
    /**
     * Set text cursor location
     */
    virtual void locate(int column, int row);
    
    /**
     * 
     */
    int lastCount;
    
    /**
     * 
     */
    int NAKCount;
    
protected:
    virtual int _putc(int value);
    virtual int _getc();
    
    /**
     * Text cursor column number
     */
    short _column;
    
    /**
     * Text cursor row number
     */
    short _row;
    
    char _fontSize;
    char _penSize;
    int _fontColor;
    
private:
    Serial s;
    DigitalOut  reset;
};

#endif