#include "mbed.h"
#include "https_request.h"
#include "network-helper.h"
#include "TCPSocket.h"
#include "NTPClient.h"
#include "HTS221Sensor.h"
#include "MbedJSONValue.h"
#include "OLEDDisplay.h"

// Name of the Device shown on Website
char* DeviceName = "ENTER_YOUR_DEVICE_NAME_HERE";

// API Token
char* token = "Bearer ENTER_API_TOKEN_HERE";

// Route to submit Temp
char* ApiRoute = "ENTER_API_ENDPOINT_HERE";

// Trusted Root Certs
const char* DST_ROOT_CA_X3 = "-----BEGIN CERTIFICATE-----\n"
    "MIIDSjCCAjKgAwIBAgIQRK+wgNajJ7qJMDmGLvhAazANBgkqhkiG9w0BAQUFADA/\n"
    "MSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT\n"
    "DkRTVCBSb290IENBIFgzMB4XDTAwMDkzMDIxMTIxOVoXDTIxMDkzMDE0MDExNVow\n"
    "PzEkMCIGA1UEChMbRGlnaXRhbCBTaWduYXR1cmUgVHJ1c3QgQ28uMRcwFQYDVQQD\n"
    "Ew5EU1QgUm9vdCBDQSBYMzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB\n"
    "AN+v6ZdQCINXtMxiZfaQguzH0yxrMMpb7NnDfcdAwRgUi+DoM3ZJKuM/IUmTrE4O\n"
    "rz5Iy2Xu/NMhD2XSKtkyj4zl93ewEnu1lcCJo6m67XMuegwGMoOifooUMM0RoOEq\n"
    "OLl5CjH9UL2AZd+3UWODyOKIYepLYYHsUmu5ouJLGiifSKOeDNoJjj4XLh7dIN9b\n"
    "xiqKqy69cK3FCxolkHRyxXtqqzTWMIn/5WgTe1QLyNau7Fqckh49ZLOMxt+/yUFw\n"
    "7BZy1SbsOFU5Q9D8/RhcQPGX69Wam40dutolucbY38EVAjqr2m7xPi71XAicPNaD\n"
    "aeQQmxkqtilX4+U9m5/wAl0CAwEAAaNCMEAwDwYDVR0TAQH/BAUwAwEB/zAOBgNV\n"
    "HQ8BAf8EBAMCAQYwHQYDVR0OBBYEFMSnsaR7LHH62+FLkHX/xBVghYkQMA0GCSqG\n"
    "SIb3DQEBBQUAA4IBAQCjGiybFwBcqR7uKGY3Or+Dxz9LwwmglSBd49lZRNI+DT69\n"
    "ikugdB/OEIKcdBodfpga3csTS7MgROSR6cz8faXbauX+5v3gTt23ADq1cEmv8uXr\n"
    "AvHRAosZy5Q6XkjEGB5YGV8eAlrwDPGxrancWYaLbumR9YbK+rlmM6pZW87ipxZz\n"
    "R8srzJmwN0jP41ZL9c8PDHIyh8bwRLtTcm1D9SZImlJnt1ir/md2cXjbDaJWFBM5\n"
    "JDGFoqgCWjBH4d1QB7wCCZAA62RjYJsWvIjJEubSfZGL+T0yjWW06XyxV3bqxbYo\n"
    "Ob8VZRzI9neWagqNdwvYkQsEjgfbKbYK7p2CNTUQ\n"
    "-----END CERTIFICATE-----\n";

// Declare LEDs
DigitalOut Led1(MBED_CONF_IOTKIT_LED1);
DigitalOut Led2(MBED_CONF_IOTKIT_LED2);
DigitalOut Led3(MBED_CONF_IOTKIT_LED3);
DigitalOut Led4(MBED_CONF_IOTKIT_LED4);

// OLED Init
OLEDDisplay oled( MBED_CONF_IOTKIT_OLED_RST, MBED_CONF_IOTKIT_OLED_SDA, MBED_CONF_IOTKIT_OLED_SCL );

// Temp Sensor Init
static DevI2C devI2c( MBED_CONF_IOTKIT_I2C_SDA, MBED_CONF_IOTKIT_I2C_SCL );
static HTS221Sensor hum_temp(&devI2c);

// Network
NetworkInterface* network;

//Global temp var
float temp;

bool InitTime() {
    oled.printf("Getting Time...");
    NTPClient ntp(network);
    time_t timestamp = ntp.get_timestamp();
    
    if (timestamp < 0) {
        printf("An error occurred when getting the time. Code: %ld\r\n", timestamp);
        return false;
    }
    else {
        set_time(timestamp);
        return true;
    }
}

bool InitializeNetwork() {
    
    oled.clear();
    oled.printf("Trying to Connect to Network...\n");
    printf("Trying to Connect to Network...\n");
    
    // Network Init
    network = connect_to_default_network_interface();
    oled.clear();

    if (!network) {
        oled.printf("Cannot connect to the network\n");
        printf("Cannot connect to the network\n");
        return false;
    }
    else {
        oled.printf("Connected to Network\n");
        printf("Connected to Network\n");
        return true;
    }
}

void PrintTemp() {
    oled.clear();
    printf("Temp: %3.2f\n", temp);
    oled.printf("Temp: %3.2f\n", temp);
}

char* SendRequest(float temperature) {

    char stringTemp[20];
    sprintf(stringTemp, "%f", temperature);
    char timeBuffer[32];
    
    time_t seconds = time(NULL);
    
    strftime(timeBuffer, 32, "%F %T", localtime(&seconds));

    HttpsRequest* post_req = new HttpsRequest(network, DST_ROOT_CA_X3, HTTP_POST, ApiRoute);
    post_req->set_header("Content-Type", "application/json");
    post_req->set_header("Authorization", token);

    char body[255] = "{\"recorded\":\"";
    strcat(body, stringTemp);
    strcat(body, "\",\"recorded_at\":\"");
    strcat(body, timeBuffer);
    strcat(body, "\",\"device\":\"");
    strcat(body, DeviceName);
    strcat(body, "\"}");
    
    printf("%s\n", body);
    
    HttpResponse* post_res = post_req->send(body, strlen(body));
    if (!post_res) {
        printf("HttpRequest failed (error code %d)\n", post_req->get_error());
    }    
    
    char* resp = (char*)post_res->get_body_as_string().c_str();
    
    delete post_req;
    printf("%s\n", resp);
    return resp;
}

void SetLEDStatusCode(char* response) {
    MbedJSONValue parser;

    parse(parser, response);
    
    int statusCode;
    
    statusCode = parser["code"].get<int>();
    
    if (statusCode == 0) {
        PrintTemp();
        Led1.write(0);
        Led2.write(0);
        Led3.write(0);
        Led4.write(0);
    }
    else if(statusCode == 1) {
        PrintTemp();
        printf("Temperature is above the limits!\n");
        oled.printf("Temperature is above the limits!\n");
        Led1.write(1);
        Led2.write(0);
        Led3.write(0);
        Led4.write(0);
    }
    else if(statusCode == 2) {
        PrintTemp();
        printf("Temperature is below the limits!\n");
        oled.printf("Temperature is below the limits!\n");
        Led1.write(0);
        Led2.write(1);
        Led3.write(0);
        Led4.write(0);
    }
    else if(statusCode == 3) {
        PrintTemp();
        printf("Undefined Error!\n");
        oled.printf("Undefined Error!\n");
        Led1.write(0);
        Led2.write(0);
        Led3.write(1);
        Led4.write(0);
    }
    else if(statusCode == 4) {
        PrintTemp();
        printf("Undefined Error!\n");
        Led1.write(0);
        Led2.write(0);
        Led3.write(0);
        Led4.write(1);
    }
}

void InitializeTempSensor() {
    
    hum_temp.init(NULL);
    hum_temp.enable();
}

int main() {
    
    if (!InitializeNetwork()) {
        return 1;
    }
    
    if (!InitTime()) {
        return 1;
    }
    
    time_t seconds = time(NULL);
    printf("\rDate & Time: \r%s", ctime(&seconds));
    
    InitializeTempSensor();
    
    wait(2);
    
    while (true) {
        oled.clear();
        hum_temp.get_temperature(&temp);
        SetLEDStatusCode(SendRequest(temp));
    }
}