/*
説明
Nucleo-F303K8とBMP180を使った気温・気圧・高度計算のサンプルプログラム

ライブラリ
https://developer.mbed.org/users/spiridion/code/BMP180/

以下ピン配置
Nucleo  BMP180
GND-----GND-------0V
+3V3----VIN
D4------SDA
D5------SCL

*/
#include "mbed.h"
#include "BMP180.h"
#define p0 1013.25f//海面気圧

DigitalOut led(PA_8);
DigitalIn flyp(PF_0);
DigitalOut cutp1(PB_0);
DigitalOut cutp2(PB_1);
BMP180  bmp(PB_7, PB_6);
Serial  dbg(PA_9, PA_10);
Timer timer;

float getAlt(float press, float temp);

#define GUDGE_WAIT_TIME     600 //[sec] 60*10
#define GUDGE_LIMIT_TIME    1800
#define CUT_ON_TIME         50 //ニクロム線の電源ON時間 [sec]
#define GUDGE_HIGHT         2 //[m]

int main() {
    led = 1;
    dbg.printf("Hello\r\n");
    cutp1 = 0;
    cutp2 = 0;
    float pressure,temperature,alt_gnd,alt_now;
    float alt[30];
    float time;
    int i = 0;   
    bmp.Initialize(64,BMP180_OSS_ULTRA_LOW_POWER);  //BMP180初期化
    alt_gnd = 0.0;
    int time_cnt = 0;
    //地上高度取得（10回の平均値）
    for(i =0; i < 10; i++){
        bmp.ReadData(&temperature,&pressure);
        alt_gnd = alt_gnd + getAlt(pressure,temperature);
    }
    alt_gnd = alt_gnd/10.0;
    dbg.printf("GND:%f",alt_gnd);
    dbg.printf("standby\r\n");
    led = 0;
    
    i=0;
    
    while(i < 10){   //フライトピン抜けるまで待つ
        if(flyp == 0){
            i++;
        }else{
            i = 0;
        }
        wait(0.5);
        dbg.printf("%d",i);
        led = !led;
    }
    
    dbg.printf("rerease\r\n");
    timer.start();
    wait(5.0);  //落ち着くまで待つ
    led = 0;
    i = 1;
    int cnt = 0;
    
    //高度が50m以下または10分たつ，になるまで着地判定はしない．
    while(1){
         wait(0.1);
         bmp.ReadData(&temperature,&pressure);
         alt_now = getAlt(pressure,temperature) - alt_gnd;
         dbg.printf("NOW:%f\r\n",alt_now);
         time = timer.read();
         
         if(alt_now < GUDGE_HIGHT){
             cnt++;
        }
        
         if(time > GUDGE_WAIT_TIME && cnt > 10){
             time_cnt ++;
             timer.reset();
             wait(0.1);
             timer.start();
        }
         if(time_cnt == 1) break;
    }
    dbg.printf("under10m\r\n");
    led = 0;
    wait(2.0);
    float land = 0.0;
    cnt = 0;
    led = 1;
    //30分たつ又は高度変化が1mいない，で着地判定
    while(1) {
        bmp.ReadData(&temperature,&pressure);
        time = timer.read();
        if(time > GUDGE_LIMIT_TIME){
             time_cnt ++;
             timer.reset();
             wait(0.1);
             timer.start();
        }
        alt[i] = getAlt(pressure,temperature);
        dbg.printf("alt[%d]:%f",i,alt[i]);
        if(i > 0){
            land = alt[i-1] - alt[i];
            if(land < 0.0f) land = land * (-1.0f);    //-を＋に
            if(land < 1.0f) cnt++;   //約1秒前との高度差が1m未満の場合，カウント
        }
        if(i == 30){    //30カウント（約30秒）でカウントリセット
            i = 0;
            cnt = 0;
        }
        dbg.printf("cnt:%d\r\n",cnt);
        if(time_cnt == 2 || cnt == 10) break; //30カウントのうち10カウント又は30分経過で着地判定
        wait(1.0);
        i ++;
    }
    timer.stop();
    //ニクロム線に10秒ずつ電流流すを繰り返す
    while(1){
        cutp1 = 1;
        //for(i = 0;i)
        led = 0;
        wait(CUT_ON_TIME);
        cutp1 = 0;
        wait(1);
        cutp2 = 1;
        led = 1;
        wait(CUT_ON_TIME);
        cutp2 = 0;
        wait(1);
        while(flyp == 1){
            dbg.printf("stop\r\n");
            wait(1);
        }
    }
}

float getAlt(float press, float temp){
    return (pow((p0/press), (1.0f/5.257f))-1.0f)*(temp+273.15f)/0.0065f;
}