/* EMW3162 implementation of NetworkInterfaceAPI
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 

#include "EMW3162Interface.h"

// Various timeouts for different EMW3162 operations
#define EMW3162_CONNECT_TIMEOUT 15000
#define EMW3162_SEND_TIMEOUT    500
#define EMW3162_RECV_TIMEOUT    0
#define EMW3162_MISC_TIMEOUT    500


// EMW3162Interface implementation
EMW3162Interface::EMW3162Interface(PinName tx, PinName rx, bool debug)
    : _emw(tx, rx, debug)
{
    memset(_ids, 0, sizeof(_ids));
    memset(_cbs, 0, sizeof(_cbs));

    _emw.attach(this, &EMW3162Interface::event);
}

int EMW3162Interface::connect()
{
    _emw.setTimeout(EMW3162_CONNECT_TIMEOUT);

    if (!_emw.startup()) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    if (!_emw.dhcp(true)) {
        return NSAPI_ERROR_DHCP_FAILURE;
    }

    if (!_emw.connect(ap_ssid, ap_pass)) {
        return NSAPI_ERROR_NO_CONNECTION;
    }

    if (!_emw.getIPAddress()) {
        return NSAPI_ERROR_DHCP_FAILURE;
    }

    return 0;
}
int EMW3162Interface::connect(const char *ssid, const char *pass, nsapi_security_t security,
                                        uint8_t channel)
{
    if (channel != 0) {
        return NSAPI_ERROR_UNSUPPORTED;
    }

    set_credentials(ssid, pass, security);
    return connect();
}

int EMW3162Interface::set_credentials(const char *ssid, const char *pass, nsapi_security_t security)
{
    memset(ap_ssid, 0, sizeof(ap_ssid));
    strncpy(ap_ssid, ssid, sizeof(ap_ssid));

    memset(ap_pass, 0, sizeof(ap_pass));
    strncpy(ap_pass, pass, sizeof(ap_pass));

    ap_sec = security;

    return 0;
}

int EMW3162Interface::set_channel(uint8_t channel)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int EMW3162Interface::disconnect()
{
    _emw.setTimeout(EMW3162_MISC_TIMEOUT);

    if (!_emw.disconnect()) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    return 0;
}

const char* EMW3162Interface::get_ip_address()
{
    return _emw.getIPAddress();
}

const char* EMW3162Interface::get_mac_address()
{
    return _emw.getMACAddress();
}

const char *EMW3162Interface::get_gateway()
{
    return NULL;
}

const char *EMW3162Interface::get_netmask()
{
    return NULL;
}

int8_t EMW3162Interface::get_rssi()
{
    return 0;
}

int EMW3162Interface::scan(WiFiAccessPoint *res, unsigned count)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

struct EMW3162_socket {
    int id;
    int socketId;
    nsapi_protocol_t proto;
    bool connected;
};

int EMW3162Interface::socket_open(void **handle, nsapi_protocol_t proto)
{
    // Look for an unused socket
    int id = -1;
 
    for (int i = 1; i < EMW3162_SOCKET_COUNT; i++) {
        if (!_ids[i]) {
            id = i;
            _ids[i] = true;
            break;
        }
    }
 
    if (id == -1) {
        return NSAPI_ERROR_NO_SOCKET;
    }
    
    struct EMW3162_socket *socket = new struct EMW3162_socket;
    if (!socket) {
        return NSAPI_ERROR_NO_SOCKET;
    }
    
    socket->id = id;
    socket->socketId = 0;
    socket->proto = proto;
    socket->connected = false;
    *handle = socket;
    return 0;
}

int EMW3162Interface::socket_close(void *handle)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;
    int err = 0;
    _emw.setTimeout(EMW3162_MISC_TIMEOUT);
 
    if (!_emw.close(socket->socketId)) {
        err = NSAPI_ERROR_DEVICE_ERROR;
    }

    _ids[socket->id] = false;
    delete socket;
    return err;
}

int EMW3162Interface::socket_bind(void *handle, const SocketAddress &address)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int EMW3162Interface::socket_listen(void *handle, int backlog)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int EMW3162Interface::socket_connect(void *handle, const SocketAddress &addr)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;
    _emw.setTimeout(EMW3162_MISC_TIMEOUT);

    const char *proto = (socket->proto == NSAPI_UDP) ? "UNICAST" : "CLIENT";
    socket -> socketId = _emw.open(proto, socket->id, addr.get_ip_address(), addr.get_port());
    if (!(socket -> socketId)) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }
    
    socket->connected = true;
    return 0;
}
    
int EMW3162Interface::socket_accept(void *handle, void **socket, SocketAddress *address)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int EMW3162Interface::socket_send(void *handle, const void *data, unsigned size)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;
    _emw.setTimeout(EMW3162_SEND_TIMEOUT);
 
    if (!_emw.send(socket->socketId, data, size)) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }
 
    return size;
}

int EMW3162Interface::socket_recv(void *handle, void *data, unsigned size)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;
    _emw.setTimeout(EMW3162_RECV_TIMEOUT);
 
    int32_t recv = _emw.recv(socket->socketId, data, size);
    if (recv < 0) {
        return NSAPI_ERROR_WOULD_BLOCK;
    }
 
    return recv;
}

int EMW3162Interface::socket_sendto(void *handle, const SocketAddress &addr, const void *data, unsigned size)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;
    if (!socket->connected) {
        int err = socket_connect(socket, addr);
        if (err < 0) {
            return err;
        }
    }
    
    return socket_send(socket, data, size);
}

int EMW3162Interface::socket_recvfrom(void *handle, SocketAddress *addr, void *data, unsigned size)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;    
    return socket_recv(socket, data, size);
}

void EMW3162Interface::socket_attach(void *handle, void (*callback)(void *), void *data)
{
    struct EMW3162_socket *socket = (struct EMW3162_socket *)handle;    
    _cbs[socket->id].callback = callback;
    _cbs[socket->id].data = data;
}

void EMW3162Interface::event() {
    for (int i = 0; i < EMW3162_SOCKET_COUNT; i++) {
        if (_cbs[i].callback) {
            _cbs[i].callback(_cbs[i].data);
        }
    }
}
