/** Class: Out_SSMulCon
 *
 * Output class for Sega Saturn's Sega Multi Controller
 *
 */

/*
注意書き・特殊操作

・入力デバイス：サイバースティックアナログモードのとき
  - サターンからはマルコンアナログモードに見える
  - G(SELECT)を押しながら上下左右スティック･･･十字キー操作
  - G(SELECT)を押しながらスロットル操作    ･･･LRキー操作

・入力デバイス：サイバースティックデジタルモードのとき
  - サターンからはマルコンデジタルモードに見える
  - DボタンがStartキーになる

・入力デバイス：ファイティングパッド6Bのとき
  - サターンからはマルコンデジタルモードに見える
  - modeキー押しながらトリガ･･･連射
  - modeキー左右･･･LRキー操作

*/

#include "mbed.h"
#include "InputStatus.h"

class Out_SSMulCon
{
public:
    /** Constructor: Out_SSMulcon
     *
     * Parameters:
     * pn_D0        - DigitalOut for D0 (SS pad connector's pin3)
     * pn_D1        - DigitalOut for D1 (pin2)
     * pn_D2        - DigitalOut for D2 (pin8)
     * pn_D3        - DigitalOut for D3 (pin7)
     * pn_TL        - InterruptIn for TL - L=DataEnable (pin4)
     * pn_TR        - InterruptIn for TR - Clock (pin5)
     * pn_TH        - DigitalOut for TH (pin6)
     * inputStatus  - Input status
     */

    Out_SSMulCon(
        PinName pn_D0, PinName pn_D1, PinName pn_D2, PinName pn_D3, 
        PinName pn_TL, PinName pn_TR, PinName pn_TH, 
        InputStatus *inputStatus
    );
     
private:
    // Private constants
    static const char OUTPUTMODE_ANALOG  = 0;
    static const char OUTPUTMODE_DIGITAL = 1;
    static const char OUTPUTMODE_6B      = 2;
    
    // mbed pins
    DigitalOut   _OUT_D0;
    DigitalOut   _OUT_D1;
    DigitalOut   _OUT_D2;
    DigitalOut   _OUT_D3;
    InterruptIn  _INTR_TL;
    InterruptIn  _INTR_TR;
    DigitalOut   _OUT_TH;

    // Variable
    InputStatus     *_InputStatus;
    char            _OutputMode;
    volatile char   _NowWriting;
    volatile char   _PhaseCounter;
    volatile char   _RapidFireValue;
    
    
    // Private Method
    void Initialize(void);
    void InitializePinStatus(void);
    void SetPinStatus(void);
    void SetD3D2D1D0(char d3, char d2, char d1, char d0);
    void SetBus(char data);

    void StrobeFallISR(void);
    void StrobeRiseISR(void);
    void ClockFallISR(void);
    void ClockRiseISR(void);
    
    void RenewOutputMode(void);



// for InputControll
public:
    void SetupInputControll(void (*startInputFunction)(void), void (*stopInputFunction)(void));
    class CDummy;
    template<class T>
    void SetupInputControll(T* inputInstance, void (T::*startInputMethod)(void), void (T::*stopInputMethod)(void))
    {
        _InputInstance    = (CDummy*) inputInstance;
        StartInputMethod = (void (CDummy::*)(void)) startInputMethod;
        StopInputMethod  = (void (CDummy::*)(void)) stopInputMethod;
    }

private:
    CDummy*         _InputInstance;
    void (CDummy::*StartInputMethod)(void);
    void (CDummy::*StopInputMethod)(void);
    void (*StartInputFunction)(void);
    void (*StopInputFunction)(void);
    void EnableInput(void);
    void DisableInput(void);

};


