/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "UbloxATCellularInterfaceExt.h"
#include "gnss.h"

// The credentials of the SIM in the board.  If PIN checking is enabled
// for your SIM card you must set this to the required PIN.
#define PIN "0000"

// Network credentials.  You should set this according to your
// network/SIM card.  For C030 boards, leave the parameters as NULL
// otherwise, if you do not know the APN for your network, you may
// either try the fairly common "internet" for the APN (and leave the
// username and password NULL), or you may leave all three as NULL and then
// a lookup will be attempted for a small number of known networks
// (see APN_db.h in mbed-os/features/netsocket/cellular/utils).
#define APN         NULL
#define USERNAME    NULL
#define PASSWORD    NULL

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// The user button
volatile bool buttonPressed = false;

static void good() {
    ledGreen = 0;
    ledBlue = 1;
    ledRed = 1;
}

static void bad() {
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
}

static void event() {
    ledBlue = 0;
    ledRed = 1;
    ledGreen = 1;
}

static void pulseEvent() {
    event();
    wait_ms(500);
    good();
}

static void ledOff() {
    ledBlue = 1;
    ledRed = 1;
    ledGreen = 1;
}

static void printCellLocateData(UbloxATCellularInterfaceExt::CellLocData *pData)
{
    char timeString[25];

    printf("Cell Locate data:\n");
    if (strftime(timeString, sizeof(timeString), "%F %T", (const tm *) &(pData->time)) > 0) {
        printf("  time:               %s\n", timeString);
    }
    printf("  longitude:          %.6f\n", pData->longitude);
    printf("  latitude:           %.6f\n", pData->latitude);
    printf("  altitude:           %d metre(s)\n", pData->altitude);
    switch (pData->sensor) {
        case UbloxATCellularInterfaceExt::CELL_LAST:
            printf("  sensor type:        last\n");
            break;
        case UbloxATCellularInterfaceExt::CELL_GNSS:
            printf("  sensor type:        GNSS\n");
            break;
        case UbloxATCellularInterfaceExt::CELL_LOCATE:
            printf("  sensor type:        Cell Locate\n");
            break;
        case UbloxATCellularInterfaceExt::CELL_HYBRID:
            printf("  sensor type:        hybrid\n");
            break;
        default:
            printf("  sensor type:        unknown\n");
            break;
    }
    printf("  uncertainty:        %d metre(s)\n", pData->uncertainty);
    printf("  speed:              %d metre(s)/second\n", pData->speed);
    printf("  direction:          %d degree(s)\n", pData->direction);
    printf("  vertical accuracy:  %d metre(s)/second\n", pData->speed);
    printf("  satellite(s) used:  %d\n", pData->svUsed);
    printf("I am here: "
           "https://maps.google.com/?q=%.5f,%.5f\n", pData->latitude, pData->longitude);       
}

static void cbButton()
{
    buttonPressed = true;
    pulseEvent();
}

/* This example program for the u-blox C030 and C027 boards instantiates
 * the UbloxAtCellularInterfaceExt to do FTP, HTTP and CellLocate operations.
 * It uses the site test.rebex.net for FTP testing and the site
 * developer.mbed.org for HTTP GET testing.
 * Progress may be monitored with a serial terminal running at 9600 baud.
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when an FTP get, HTTP get or CellLocate
 * operation is completed and turn red if there is a failure.
 */

int main()
{
    UbloxATCellularInterfaceExt *interface = new UbloxATCellularInterfaceExt();
    // If you need to debug the cellular interface, comment out the
    // instantiation above and uncomment the one below.
//    UbloxATCellularInterfaceExt *interface = new UbloxATCellularInterfaceExt(MDMTXD, MDMRXD,
//                                                                             MBED_CONF_UBLOX_CELL_BAUD_RATE,
//                                                                             true);
    UbloxATCellularInterfaceExt::Error *err;
    UbloxATCellularInterfaceExt::CellLocData data;
    char buf[1024];
    int httpProfile;
    int numRes;
    GnssSerial gnssSerial; // This needed purely to power on the GNSS chip in 
                           // order that Cell Locate on the module can use it
#ifdef TARGET_UBLOX_C027
    // No user button on C027
    InterruptIn userButton(NC);
#else
    InterruptIn userButton(SW0);
#endif
    
    // Attach a function to the user button
    userButton.rise(&cbButton);
    
    // Power up GNSS to assist with the Cell Locate bit
    gnssSerial.init();
    good();
    printf("Starting up, please wait up to 180 seconds for network registration to complete...\n");
    if (interface->init(PIN)) {
        pulseEvent();
        interface->set_credentials(APN, USERNAME, PASSWORD);
        printf("Registered, connecting to the packet network...\n");
        for (int x = 0; interface->connect() != 0; x++) {
            if (x > 0) {
                bad();
                printf("Retrying (have you checked that an antenna is plugged in and your APN is correct?)...\n");
            }
        }
        pulseEvent();

        // FTP OPERATIONS
        // Reset FTP parameters to default then set things up
        printf("=== FTP ===\n");
        interface->ftpResetPar();
        interface->ftpSetTimeout(60000);
        interface->ftpSetPar(UbloxATCellularInterfaceExt::FTP_SERVER_NAME, "test.rebex.net");
        interface->ftpSetPar(UbloxATCellularInterfaceExt::FTP_USER_NAME, "demo");
        interface->ftpSetPar(UbloxATCellularInterfaceExt::FTP_PASSWORD, "password");
        interface->ftpSetPar(UbloxATCellularInterfaceExt::FTP_MODE, "1");

        // Log into the FTP server
        printf("Logging into FTP server \"test.rebex.net\"...\n");
        err = interface->ftpCommand(UbloxATCellularInterfaceExt::FTP_LOGIN);
        if (err == NULL) {
            pulseEvent();
             // Get a directory listing from the server
             if (interface->ftpCommand(UbloxATCellularInterfaceExt::FTP_LS,
                                      NULL, NULL, buf, sizeof (buf)) == NULL) {
                pulseEvent();
                printf ("Directory listing of FTP server:\n"
                        "--------------------------------\n%s"
                        "--------------------------------\n", buf);
             }
             
             // FTP GET a file known to be on test.rebex.net into the module file system,
             // making sure it's not there to begin with
             interface->delFile("readme.txt");
             if (interface->ftpCommand(UbloxATCellularInterfaceExt::FTP_GET_FILE, "readme.txt") == NULL) {
                pulseEvent();
                // Read the file from the module file system into buf
                if (interface->readFile("readme.txt", buf, sizeof (buf))) {
                    printf("FTP GET of file \"readme.txt\" completed.  The file contained:\n"
                           "------------------------------------------------------------\n%s"
                           "------------------------------------------------------------\n", buf);
                }
             }            
        } else {
            bad();
            printf ("Unable to log in, error class %d, error code %d.\n", err->eClass, err->eCode);
        }
        
        // HTTP OPERATIONS
        // Set up HTTP parameters
        printf("=== HTTP ===\n");
        printf("Performing HTTP GET on \"developer.mbed.org\"...\n");
        httpProfile = interface->httpAllocProfile();
        interface->httpSetTimeout(httpProfile, 30000);
        interface->httpSetPar(httpProfile, UbloxATCellularInterfaceExt::HTTP_SERVER_NAME, "developer.mbed.org");
        
        // Do the HTTP command
        err = interface->httpCommand(httpProfile, UbloxATCellularInterfaceExt::HTTP_GET,
                                    "/media/uploads/mbed_official/hello.txt",
                                     NULL, NULL, 0, NULL,
                                     buf, sizeof (buf));
        if (err == NULL) {
            pulseEvent();
            printf("HTTP GET of \"/media/uploads/mbed_official/hello.txt\" completed.  The response contained:\n"
                   "----------------------------------------------------------------------------------------\n%s"
                   "----------------------------------------------------------------------------------------\n", buf);
        } else {
            bad();
            printf("Unable to get \"/media/uploads/mbed_official/hello.txt\" from \"developer.mbed.org\", "
                   "error class %d, error code %d.\n", err->eClass, err->eCode);
        }

        // CELL LOCATE OPERATIONS (in a loop)
        printf("=== Cell Locate ===\n");
        printf("Sending Cell Locate requests in a loop (until the user button is pressed on C030 or forever on C027)...\n");
        while (!buttonPressed) {
            interface->cellLocSrvUdp();
            interface->cellLocConfig(1); // Deep scan mode
            printf("Sending Cell Locate request...\n");
            if (interface->cellLocRequest(UbloxATCellularInterfaceExt::CELL_HYBRID, 10, 100,
                                         (UbloxATCellularInterfaceExt::CellRespType) 1, 1)) {
                // Wait for the response
                numRes = 0;
                for (int x = 0; (numRes == 0) && (x < 10); x++) {
                    numRes = interface->cellLocGetRes();
                }
                
                if (numRes > 0) {
                    interface->cellLocGetData(&data);
                    if (data.validData) {
                        pulseEvent();
                        printCellLocateData(&data);
                    }
                } else {
                    bad();
                    printf("No response from Cell Locate server.\n");
                }
            }
            wait_ms(5000);
#ifndef TARGET_UBLOX_C027
            printf("[Checking if user button has been pressed]\n");
#endif
        }
        
        pulseEvent();
        printf("User button was pressed, stopping...\n");
        gnssSerial.powerOff();
        interface->disconnect();
        interface->deinit();
        ledOff();
        printf("Stopped.\n");
    } else {
        bad();
        printf("Unable to initialise the interface.\n");
    }
}

// End Of File