/**
* @author Samuel Mokrani
*
* @section LICENSE
*
* Copyright (c) 2011 mbed
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*
* @section DESCRIPTION
*    Simple websocket client
*
*/ 

#ifndef WEBSOCKET_H
#define WEBSOCKET_H

#include "mbed.h"
#include "Wifly.h"
#include <string>


/** Websocket client Class. 
 *
 * Warning: you must use a wifi module (Wifly RN131-C)
 *
 * Example (wifi network):
 * @code
 * #include "mbed.h"
 * #include "Wifly.h"
 * #include "Websocket.h"
 * 
 * Serial pc(USBTX, USBRX);
 * Wifly * wifly;
 * Websocket * ws;
 *
 * int main()
 * {
 *   wifly = new Wifly(p9, p10, p20, "network", "password", true);
 *   ws = new Websocket("ws://ip_domain/path", wifly);
 *   
 *   if(wifly->join())
 *   {
 *       if(ws->connect())
 *       {
 *           pc.printf("ws connected\r\n");
 *           while(1)
 *           {
 *               wait(0.1);
 *               ws->send("test");
 *           }
 *       }
 *       else
 *           pc.printf("ws not connected\r\n");
 *   }
 *   else
 *       pc.printf("join network failed\r\n");
 *       
 * }
 * @endcode
 *
 *
 */
class Websocket
{
    public:
        /**
        * Constructor
        *
        * @param url The Websocket url in the form "ws://ip_domain[:port]/path"  (by default: port = 80)
        * @param wifi pointer to a wifi module (the communication will be establish by this module)
        */
        Websocket(char * url, Wifly * wifi);
        
        
        /**
        * Connect to the websocket url
        *
        *@return true if the connection is established, false otherwise
        */
        bool connect();
        
        /**
        * Send "str" according to the websocket format: 00 str ff
        *
        * @param str string to be sent
        */
        void send(char * str);
        
        /**
        * Read a websocket message
        *
        * @param message pointer to the string to be read (null if drop frame)
        *
        * @return true if a string has been read, false otherwise
        */
        bool read(char * message);
        
        /**
        * To see if there is a websocket connection active
        *
        * @return true if there is a connection active
        */
        bool connected();
        
        /**
        * Close the websocket connection
        *
        * @return true if the connection has been closed, false otherwise
        */
        bool close();
        
        /**
        * Accessor: get path from the websocket url
        *
        * @return path
        */
        std::string getPath();
        
        enum Type {
            WIF,
            ETH
        };
        
    
    private:
    
        
        void fillFields(char * url);
        
        std::string ip_domain;
        std::string path;
        std::string port;
        
        Wifly * wifi;
        Type netif;
        

};

#endif