#ifndef CDMS_SD_INLCUDED
#define CDMS_SD_INLCUDED

//SPI spi(PTE1, PTE3, PTE2);      // MOSI,MISO, CLOCK microcontroller(in order)
//DigitalOut cs_sd(PTE22);

//Serial sd1(USBTX,USBRX);


#define SD_COMMAND_TIMEOUT 325

#define SD_DBG             1

#define R1_IDLE_STATE           (1 << 0)
#define R1_ERASE_RESET          (1 << 1)
#define R1_ILLEGAL_COMMAND      (1 << 2)
#define R1_COM_CRC_ERROR        (1 << 3)
#define R1_ERASE_SEQUENCE_ERROR (1 << 4)
#define R1_ADDRESS_ERROR        (1 << 5)
#define R1_PARAMETER_ERROR      (1 << 6)

uint32_t SD_DATABLOCK_START[] = {0, 1001, 11001, 21001, 31001, 41001};
uint32_t SD_DATABLOCK_END[] = {0, 11000, 21000, 31000, 41000, 51000};
uint32_t SD_MNG_SECT=80000;

extern uint8_t SD_INIT_FLAGS;

int initialise_card();
int initialise_card_v1();
int initialise_card_v2();
int disk_write(const uint8_t *, uint64_t);
int disk_read(uint8_t *, uint64_t);
int disk_erase(int,int);
int disk_read_statusbits(uint8_t *);

void FCTN_SD_MNGR();
int INCREMENT_SD_LIB(uint8_t);


int cmd(int, int);
int cmd58();
int cmdx(int, int);
int cmd8();
int read(uint8_t*, uint32_t );
int write(const uint8_t*, uint32_t );
static uint32_t ext_bits(unsigned char *, int , int );
int SD_WRITE(uint8_t*,uint32_t,uint8_t);
int FCTN_CDMS_SD_INIT();
uint8_t SD_READ(uint8_t*,uint32_t,uint8_t);
#define SDCARD_FAIL 4
#define SDCARD_V1   1
#define SDCARD_V2   2
#define SDCARD_V2HC 3

int cdv;
uint64_t sd_sectors();
uint64_t sectors;

int FCTN_CDMS_SD_INIT()
{
    int i = initialise_card();
    if( i == 4)
        return 4;
    debug_if(SD_DBG, "init card = %d\n", i);
    sectors = sd_sectors();

    // Set block length to 512 (CMD16)
    if (cmd(16, 512) != 0) {
        debug("\rSet 512-byte block timed out\r\n");
        return 1;
    } else {
        //printf("\rDisk initialization successfull\r\n");
    }
    SD_STATUS = DEVICE_POWERED;
    spi.frequency(1000000); // Set to 1MHz for data transfer
    return 0;
}

void  FCTN_SD_MNGR()
{
    uint32_t fsc;
    uint32_t fsc_old;
    uint8_t buffer[512];
    int b;
    if(SD_STATUS == DEVICE_POWERED) {
        b=disk_read(buffer, SD_MNG_SECT);

        for(int i = 0; i < 5; i++ ) {
            fsc= (uint32_t)(buffer[0 + i*8]<<24)+(uint32_t)(buffer[1 + i*8]<<16)+(uint32_t)(buffer[2 + i*8]<<8)+(uint32_t)buffer[3 + i*8];
            fsc_old= (uint32_t)(buffer[4 + i*8]<<24)+(uint32_t)(buffer[5 + i*8]<<16)+(uint32_t)(buffer[6 + i*8]<<8)+(uint32_t)buffer[7 + i*8];
            FSC_CURRENT[1 + i] = fsc;
            FSC_OLD[1 + i] = fsc_old;
        }
    }
}

int INCREMENT_SD_LIB(uint8_t sid)
{
    uint32_t fsc;
    uint32_t fsc_old;
    int i;
    uint8_t buffer[512];

    disk_read(buffer,SD_MNG_SECT);
    SD_MNG_SECT += SD_LIB_WRITES/(int)0xFFFF;
    if(SD_MNG_SECT != SD_LIB_BLK_CURRENT) {
        SD_LIB_BLK_CURRENT = SD_MNG_SECT;
        WRITE_TO_EEPROM(16, (int32_t)SD_LIB_BLK_CURRENT);
        //FCTN_CDMS_WR_FLASH(16,SD_LIB_BLK_CURRENT);
    }
    SD_LIB_WRITES = SD_LIB_WRITES%(int)0xFFFF;
    disk_write(buffer,SD_MNG_SECT);

    disk_read(buffer,SD_MNG_SECT);
    if(sid>=0x01 && sid <=0x05)
    {
        fsc=(uint32_t)(buffer[0 + ((sid-1)*8)]<<24)+(uint32_t)(buffer[1 + ((sid-1)*8)]<<16)+(uint32_t)(buffer[2 + ((sid-1)*8)]<<8)+(uint32_t)buffer[3 + ((sid-1)*8)];
        fsc_old=(uint32_t)(buffer[4 + ((sid-1)*8)]<<24)+(uint32_t)(buffer[5 + ((sid-1)*8)]<<16)+(uint32_t)(buffer[6 + ((sid-1)*8)]<<8)+(uint32_t)buffer[7 + ((sid-1)*8)];
        fsc++;
        buffer[0 + ((sid-1)*8)]=(uint8_t) (fsc>>24 & 0xFF);
        buffer[1 + ((sid-1)*8)]=(uint8_t) (fsc>>16 & 0xFF);
        buffer[2 + ((sid-1)*8)]=(uint8_t) (fsc>>8 & 0xFF);
        buffer[3 + ((sid-1)*8)]=(uint8_t) (fsc & 0xFF);
        if(fsc > SD_DATABLOCK_END[sid]-SD_DATABLOCK_START[sid]+1) {
            fsc_old = fsc_old+1;
            buffer[4 + ((sid-1)*8)]=(uint8_t) (fsc_old>>24 & 0xFF);
            buffer[5 + ((sid-1)*8)]=(uint8_t) (fsc_old>>16 & 0xFF);
            buffer[6 + ((sid-1)*8)]=(uint8_t) (fsc_old>>8 & 0xFF);
            buffer[7 + ((sid-1)*8)]=(uint8_t) (fsc_old & 0xFF);
        }

        i = disk_write(buffer,SD_MNG_SECT);
        if(i == 0) {
            FSC_CURRENT[sid] = fsc;
            FSC_OLD[sid] = fsc_old;
            return i;
        }
    }
    return -1;
}


int SD_WRITE(uint8_t* buffer,uint32_t fsc,uint8_t sid)
{
    uint32_t block_number;
    int result = 10;
    if(SD_STATUS == DEVICE_POWERED) {
        if(sid==0x01) {
            block_number=SD_DATABLOCK_START[1]+(fsc%(SD_DATABLOCK_END[1]-SD_DATABLOCK_START[1]+1));
            //block_number=SD_DATABLOCK_START[1]+fsc;
            result= disk_write(buffer,block_number);
            if(result == 0) {
                if(INCREMENT_SD_LIB(sid) == 0)
                    SD_LIB_WRITES++;
            }
            return result;
        }
        if(sid==0x02) {
            block_number= SD_DATABLOCK_START[2]+(fsc%(SD_DATABLOCK_END[2] - SD_DATABLOCK_START[2]+1));
            //block_number= SD_DATABLOCK_START[2]+fsc;
            result= disk_write(buffer,block_number);
            if(result == 0) {
                if(INCREMENT_SD_LIB(sid) == 0)
                    SD_LIB_WRITES++;
            }
            return result;
        }
        if(sid==0x03) {
            block_number= SD_DATABLOCK_START[3] +(fsc%(SD_DATABLOCK_END[3] - SD_DATABLOCK_START[3] +1));
            //block_number= SD_DATABLOCK_START[3] +fsc;
            result= disk_write(buffer,block_number);
            if(result == 0) {
                if(INCREMENT_SD_LIB(sid) == 0)
                    SD_LIB_WRITES++;
            }
            return result;
        }
        if(sid==0x04) {
            block_number=SD_DATABLOCK_START[4] +(fsc%(SD_DATABLOCK_END[4] - SD_DATABLOCK_START[4] +1));
            //block_number=SD_DATABLOCK_START[4] +fsc;
            result= disk_write(buffer,block_number);
            if(result == 0) {
                if(INCREMENT_SD_LIB(sid) == 0)
                    SD_LIB_WRITES++;
            }
            return result;
        }
        if(sid==0x05) {
            block_number= SD_DATABLOCK_START[5] +(fsc%(SD_DATABLOCK_START[5] - SD_DATABLOCK_START[5] +1));
            //block_number= SD_DATABLOCK_START[5] +fsc;
            result= disk_write(buffer,block_number);
            if(result == 0) {
                if(INCREMENT_SD_LIB(sid) == 0)
                    SD_LIB_WRITES++;
            }
            return result;
        }
    }
    // return 1;
}

uint8_t SD_READ(uint8_t* buffer,uint32_t fsc,uint8_t sid)
{
    FCTN_SD_MNGR();
    uint32_t block_number;
    int result;
    //if(SD_SW_EN_DS == 1)
    //    return 0x89;
    if(sid==0x01) {
        if(!(FSC_OLD[1]<=fsc && fsc<=FSC_CURRENT[1])) {
            return 0x86;
        }
        block_number=SD_DATABLOCK_START[1]+(fsc%(SD_DATABLOCK_END[1]-SD_DATABLOCK_START[1]+1));
        result= disk_read(buffer,block_number);
    } else if(sid==0x02) {
        if(!(FSC_OLD[2]<=fsc && fsc<=FSC_CURRENT[2])) {
            return 0x86;
        }
        block_number= SD_DATABLOCK_START[2]+(fsc%(SD_DATABLOCK_END[2] - SD_DATABLOCK_START[2]+1));
        result= disk_read(buffer,block_number);
    } else if(sid==0x03) {
        if(!(FSC_OLD[3]<=fsc && fsc<=FSC_CURRENT[3])) {
            return 0x86;
        }
        block_number= SD_DATABLOCK_START[3] +(fsc%(SD_DATABLOCK_END[3] - SD_DATABLOCK_START[3] +1));
        result= disk_read(buffer,block_number);
    } else if(sid==0x04) {
        if(!(FSC_OLD[4]<=fsc && fsc<=FSC_CURRENT[4])) {
            return 0x86;
        }
        block_number=SD_DATABLOCK_START[4] +(fsc%(SD_DATABLOCK_END[4] - SD_DATABLOCK_START[4] +1));
        result= disk_read(buffer,block_number);
    } else if(sid==0x05) {
        if(!(FSC_OLD[5]<=fsc && fsc<=FSC_CURRENT[5])) {
            return 0x86;
        }
        block_number= SD_DATABLOCK_START[5] +(fsc%(SD_DATABLOCK_START[5] - SD_DATABLOCK_START[5] +1));
        result= disk_read(buffer,block_number);
    } else {
        return 0x02;
    }
    if(result == 0)
        return 0xA0;
    else
        return 0x88;
    return 0xA0;
}


int initialise_card()
{
    // Set to 100kHz for initialisation, and clock card with cs_sd = 1
    spi.frequency(100000);           // changed on 31 12 2015 to 1 MHz
    cs_sd = 1;
    for (int i = 0; i < 16; i++) {
        spi.write(0xFF);
    }
    uint8_t R1_response = cmd(0,0);
    gPC.printf("0x%02X",R1_response);
    // send CMD0, should return with all zeros except IDLE STATE set (bit 0)
    if (R1_response != R1_IDLE_STATE) {
        debug("No disk, or could not put SD card in to spi idle state\r\n");
        return SDCARD_FAIL;
    } else
        gPC.puts("SD Card is in IDLE state\n\r");

    // send CMD8 to determine whther it is ver 2.x
    int r = cmd8();
    if (r == R1_IDLE_STATE) {
        gPC.puts("Entering V2\r");
        int q =  initialise_card_v2();
        return q;

    } else if (r == (R1_IDLE_STATE | R1_ILLEGAL_COMMAND)) {
        gPC.puts("Entering V1");
        return initialise_card_v1();

    } else {
        debug("\rNot in idle state after sending CMD8 (not an SD card?)\r\n");
        return SDCARD_FAIL;
    }
}

int initialise_card_v1()
{
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        cmd(55, 0);
        if (cmd(41, 0) == 0) {
            gPC.puts("\rv1 initialization successfull\r\n");
            cdv = 512;
            debug_if(SD_DBG, "\n\rInit: SEDCARD_V1\n\r");
            return SDCARD_V1;
        }
    }

    debug("\rTimeout waiting for v1.x card\r\n");
    return SDCARD_FAIL;
}


int initialise_card_v2()
{
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        wait_ms(50);
        cmd58();
        cmd(55, 0);
        if (cmd(41, 0x40000000) == 0) {
            if (DEBUG)
                gPC.puts("\rv2 initialization successfull\r\n");
            cmd58();
            debug_if(SD_DBG, "\n\rInit: SDCARD_V2\n\r");
            cdv = 1;
            FCTN_SD_MNGR();
            return SDCARD_V2;
        }
    }

    debug("\rTimeout waiting for v2.x card\r\n");
    return SDCARD_FAIL;
}

int cmd(int cmd, int arg)
{
    cs_sd = 0;

    // send a command
    spi.write(0x40 | cmd);
    spi.write(arg >> 24);
    spi.write(arg >> 16);
    spi.write(arg >> 8);
    spi.write(arg >> 0);
    spi.write(0x95);

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        int response = spi.write(0xFF);
        if (!(response & 0x80)) {
            cs_sd = 1;
            spi.write(0xFF);
            return response;
        }
    }
    cs_sd = 1;
    spi.write(0xFF);
    return -1; // timeout
}


int cmd58()
{
    cs_sd = 0;
    int arg = 0;

    // send a command
    spi.write(0x40 | 58);
    spi.write(arg >> 24);
    spi.write(arg >> 16);
    spi.write(arg >> 8);
    spi.write(arg >> 0);
    spi.write(0x95);

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        int response = spi.write(0xFF);
        if (!(response & 0x80)) {
            int ocr = spi.write(0xFF) << 24;
            ocr |= spi.write(0xFF) << 16;
            ocr |= spi.write(0xFF) << 8;
            ocr |= spi.write(0xFF) << 0;
            cs_sd = 1;
            spi.write(0xFF);
            return response;
        }
    }
    cs_sd = 1;
    spi.write(0xFF);
    return -1; // timeout
}


int cmd8()
{
    cs_sd = 0;

    // send a command
    spi.write(0x40 | 8); // CMD8
    spi.write(0x00);     // reserved
    spi.write(0x00);     // reserved
    spi.write(0x01);     // 3.3v
    spi.write(0xAA);     // check pattern
    spi.write(0x87);     // crc

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT * 1000; i++) {
        char response[5];
        response[0] = spi.write(0xFF);
        if (!(response[0] & 0x80)) {
            for (int j = 1; j < 5; j++) {
                response[i] = spi.write(0xFF);
            }
            cs_sd = 1;
            spi.write(0xFF);
            return response[0];
        }
    }
    cs_sd = 1;
    spi.write(0xFF);
    return -1; // timeout
}

uint64_t sd_sectors()
{
    uint32_t c_size, c_size_mult, read_bl_len;
    uint32_t block_len, mult, blocknr, capacity;
    uint32_t hc_c_size;
    uint64_t blocks;

    // CMD9, Response R2 (R1 byte + 16-byte block read)
    if (cmdx(9, 0) != 0) {
        debug("\rDidn't get a response from the disk\n");
        return 0;
    }

    uint8_t cs_sdd[16];
    if (read(cs_sdd, 16) != 0) {
        debug("\rCouldn't read cs_sdd response from disk\n");
        return 0;
    }

    // cs_sdd_structure : cs_sdd[127:126]
    // c_size        : cs_sdd[73:62]
    // c_size_mult   : cs_sdd[49:47]
    // read_bl_len   : cs_sdd[83:80] - the *maximum* read block length

    int cs_sdd_structure = ext_bits(cs_sdd, 127, 126);

    switch (cs_sdd_structure) {
        case 0:
            cdv = 512;
            c_size = ext_bits(cs_sdd, 73, 62);
            c_size_mult = ext_bits(cs_sdd, 49, 47);
            read_bl_len = ext_bits(cs_sdd, 83, 80);

            block_len = 1 << read_bl_len;
            mult = 1 << (c_size_mult + 2);
            blocknr = (c_size + 1) * mult;
            capacity = blocknr * block_len;
            blocks = capacity / 512;
            debug_if(SD_DBG, "\n\rSDCard\n\rc_size: %d \n\rcapacity: %ld \n\rsectors: %lld\n\r", c_size, capacity, blocks);
            break;

        case 1:
            cdv = 1;
            hc_c_size = ext_bits(cs_sdd, 63, 48);
            blocks = (hc_c_size+1)*1024;
            debug_if(SD_DBG, "\n\rSDHC Card \n\rhc_c_size: %d\n\rcapacity: %lld \n\rsectors: %lld\n\r", hc_c_size, blocks*512, blocks);
            break;

        default:
            debug("cs_sdD struct unsupported\r\n");
            return 0;
    };
    return blocks;
}

int cmdx(int cmd, int arg)
{
    cs_sd = 0;

    // send a command
    spi.write(0x40 | cmd);
    spi.write(arg >> 24);
    spi.write(arg >> 16);
    spi.write(arg >> 8);
    spi.write(arg >> 0);
    spi.write(0x95);

    // wait for the repsonse (response[7] == 0)
    for (int i = 0; i < SD_COMMAND_TIMEOUT; i++) {
        int response = spi.write(0xFF);
        if (!(response & 0x80)) {
            return response;
        }
    }
    cs_sd = 1;
    spi.write(0xFF);
    return 1; // timeout
}

static uint32_t ext_bits(unsigned char *data, int msb, int lsb)
{
    uint32_t bits = 0;
    uint32_t size = 1 + msb - lsb;
    for (int i = 0; i < size; i++) {
        uint32_t position = lsb + i;
        uint32_t byte = 15 - (position >> 3);
        uint32_t bit = position & 0x7;
        uint32_t value = (data[byte] >> bit) & 1;
        bits |= value << i;
    }
    return bits;
}

int disk_write(const uint8_t *buffer, uint64_t block_number)

{
    // set write address for single block (CMD24)
    if (cmd(24, block_number * cdv) != 0) {
        CDMS_WR_SD_FAULT_COUNTER++;
        return 1;
    }

    uint64_t temp;
    int r = write(buffer, 512);
    if(r == 0 ) {
        temp = FCTN_CDMS_RD_RTC();
        TIME_LATEST_SD_WR = temp >> 7; //corrected by samp:TIME_LATEST_SD_WRD = temp >> 7;
    }
    return  r;
}

int write(const uint8_t*buffer, uint32_t length)
{
    cs_sd = 0;

    // indicate start of block
    spi.write(0xFE);

    // write the data
    for (int i = 0; i < length; i++) {
        spi.write(buffer[i]);
    }

    // write the checksum
    spi.write(0xFF);
    spi.write(0xFF);

    // check the response token
    if ((spi.write(0xFF) & 0x1F) != 0x05) {
        cs_sd = 1;
        spi.write(0xFF);
        CDMS_WR_SD_FAULT_COUNTER++;
        return 1;
    }

    // wait for write to finish
    while (spi.write(0xFF) == 0);

    cs_sd = 1;
    spi.write(0xFF);
    return 0;
}

int disk_read(uint8_t *buffer, uint64_t block_number)
{
    // set read address for single block (CMD17)
    if (cmd(17, block_number * cdv) != 0) {
        SD_RD_ERROR = 1;
        return 1;
    }

    // receive the data
    read(buffer, 512);
    uint64_t temp = FCTN_CDMS_RD_RTC();
    TIME_LATEST_SD_RD = temp >> 7;
    return 0;
}

int read(uint8_t *buffer, uint32_t length)
{
    cs_sd = 0;

    // read until start byte (0xFF)
    while (spi.write(0xFF) != 0xFE);

    // read data
    for (int i = 0; i < length; i++) {
        buffer[i] = spi.write(0xFF);
    }
    spi.write(0xFF); // checksum
    spi.write(0xFF);

    cs_sd = 1;
    spi.write(0xFF);
    return 0;
}

int disk_erase(int startBlock, int totalBlocks)
{
    if(cmd(32, startBlock * cdv) != 0) {
        return 1;
    }
    if (cmd(33, (startBlock+totalBlocks-1) * cdv) != 0) {
        return 1;
    }
    if (cmd(38,0) != 0) {
        return 1;
    }

    return 0; //normal return
}

int disk_read_statusbits(uint8_t *buffer)
{
    if (cmd(17, 0) != 0) {
        SD_RD_ERROR = 1;
        return -1;
    }

    // receive the data
    return read(buffer,64);
}

#endif