/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __BLE_ENVIRONMENTAL_SERVICE_H__
#define __BLE_ENVIRONMENTAL_SERVICE_H__

#include "ble/BLE.h"

/**
* @class EnvironmentalService
* @brief BLE Environmental Service. This service provides temperature, humidity and pressure measurement.
* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.environmental_sensing.xml
* Temperature: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.temperature.xml
* Humidity: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.humidity.xml
* Pressure: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.pressure.xml
*/
class EnvironmentalService {
public:
    typedef int16_t  TemperatureType_t;
    typedef uint16_t HumidityType_t;
    typedef uint32_t PressureType_t;
    typedef uint8_t  TimeType_t;
    typedef uint8_t writeValue_t[10];
    typedef uint8_t readValue_t[10];
    typedef int16_t Magneto_t[3];
    typedef int16_t ACCELERO_t[3];
    typedef float   Gyro_t[3];
    
    const static uint16_t READ_SERVICE_CHARACTERISTIC_UUID = 0xA001;
    const static uint16_t WRITE_SERVICE_CHARACTERISTIC_UUID = 0xA002;
    const static uint16_t MAGNETO_SERVICE_CHARACTERISTIC_UUID = 0xA003;
    const static uint16_t GYRO_SERVICE_CHARACTERISTIC_UUID = 0xA004;
    const static uint16_t ACCELERO_SERVICE_CHARACTERISTIC_UUID = 0xA005;

    /**
     * @brief   EnvironmentalService constructor.
     * @param   ble Reference to BLE device.
     * @param   temperature_en Enable this characteristic.
     * @param   humidity_en Enable this characteristic.
     * @param   pressure_en Enable this characteristic.
     */
    EnvironmentalService(BLE& _ble) :
        ble(_ble),
        temperatureCharacteristic(GattCharacteristic::UUID_TEMPERATURE_CHAR, &temperature),
        humidityCharacteristic(GattCharacteristic::UUID_HUMIDITY_CHAR, &humidity),
        pressureCharacteristic(GattCharacteristic::UUID_PRESSURE_CHAR, &pressure),
        timeCharacteristic(GattCharacteristic::UUID_CURRENT_TIME_CHAR , &timeNow),
        magnetoCharacteristic(MAGNETO_SERVICE_CHARACTERISTIC_UUID, &magneto),
        acceleroCharacteristic(ACCELERO_SERVICE_CHARACTERISTIC_UUID, &accelero),
        gyroCharacteristic(GYRO_SERVICE_CHARACTERISTIC_UUID, &gyro),
        readState(READ_SERVICE_CHARACTERISTIC_UUID, &readValue)
        
    {
        static bool serviceAdded = false; /* We should only ever need to add the information service once. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = { &humidityCharacteristic,
                                            &pressureCharacteristic,
                                            &temperatureCharacteristic,
                                            &timeCharacteristic,
                                            &readState,
                                            &magnetoCharacteristic,
                                            &acceleroCharacteristic,
                                            &gyroCharacteristic };
        GattService environmentalService(GattService::UUID_ENVIRONMENTAL_SERVICE, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.gattServer().addService(environmentalService);
        serviceAdded = true;
    }

    /**
     * @brief   Update humidity characteristic.
     * @param   newHumidityVal New humidity measurement.
     */
    void updateHumidity(HumidityType_t newHumidityVal)
    {
        humidity = (HumidityType_t) (newHumidityVal*100);
        ble.gattServer().write(humidityCharacteristic.getValueHandle(), (uint8_t *) &humidity, sizeof(HumidityType_t));
    }
    
    void updateTime(TimeType_t newTimeVal)
    {
        timeNow = (TimeType_t) newTimeVal;
        ble.gattServer().write(timeCharacteristic.getValueHandle(), (uint8_t *) &timeNow, sizeof(newTimeVal));
    }

    /**
     * @brief   Update pressure characteristic.
     * @param   newPressureVal New pressure measurement.
     */
    void updatePressure(PressureType_t newPressureVal)
    {
        pressure = (PressureType_t) (newPressureVal * 10);
        ble.gattServer().write(pressureCharacteristic.getValueHandle(), (uint8_t *) &pressure, sizeof(PressureType_t));
    }

    /**
     * @brief   Update temperature characteristic.
     * @param   newTemperatureVal New temperature measurement.
     */
    void updateTemperature(float newTemperatureVal)
    {
        temperature = (TemperatureType_t) (newTemperatureVal * 100);
        ble.gattServer().write(temperatureCharacteristic.getValueHandle(), (uint8_t *) &temperature, sizeof(TemperatureType_t));
    }
    
    void updateMagneto(Magneto_t newMagnetoVal)
    {
        ble.gattServer().write(magnetoCharacteristic.getValueHandle(), (uint8_t *) &magneto[0], sizeof(int16_t));
        ble.gattServer().write(magnetoCharacteristic.getValueHandle(), (uint8_t *) &magneto[1], sizeof(int16_t));
        ble.gattServer().write(magnetoCharacteristic.getValueHandle(), (uint8_t *) &magneto[2], sizeof(int16_t));
    }
    
    void updateAccelero(ACCELERO_t newAcceleroVal)
    {
        ble.gattServer().write(acceleroCharacteristic.getValueHandle(), (uint8_t *) &accelero[0], sizeof(int16_t));
        ble.gattServer().write(acceleroCharacteristic.getValueHandle(), (uint8_t *) &accelero[1], sizeof(int16_t));
        ble.gattServer().write(acceleroCharacteristic.getValueHandle(), (uint8_t *) &accelero[2], sizeof(int16_t));
    }
    
    void updateGyro(Gyro_t newGyroVal)
    {
        ble.gattServer().write(gyroCharacteristic.getValueHandle(), (uint8_t *) &gyro[0], sizeof(float));
        ble.gattServer().write(gyroCharacteristic.getValueHandle(), (uint8_t *) &gyro[1], sizeof(float));
        ble.gattServer().write(gyroCharacteristic.getValueHandle(),  (uint8_t *)  &gyro[2],  sizeof(float));
    }
    
    GattAttribute::Handle_t getValueHandle() const
    {
        return readState.getValueHandle();
    }
    
    
    
    
private:
    BLE& ble;

    TemperatureType_t temperature;
    HumidityType_t    humidity;
    PressureType_t    pressure;
    TimeType_t        timeNow;
    writeValue_t writeValue;
    readValue_t readValue;
    Magneto_t   magneto;
    ACCELERO_t   accelero;
    Gyro_t      gyro;

    ReadOnlyGattCharacteristic<TemperatureType_t> temperatureCharacteristic;
    ReadOnlyGattCharacteristic<HumidityType_t>    humidityCharacteristic;
    ReadOnlyGattCharacteristic<PressureType_t>    pressureCharacteristic;
    ReadOnlyGattCharacteristic<TimeType_t>        timeCharacteristic;
    ReadOnlyGattCharacteristic<readValue_t> readState;
    ReadOnlyGattCharacteristic<Magneto_t> magnetoCharacteristic;
    ReadOnlyGattCharacteristic<ACCELERO_t> acceleroCharacteristic;
    ReadOnlyGattCharacteristic<Gyro_t> gyroCharacteristic;
};

#endif /* #ifndef __BLE_ENVIRONMENTAL_SERVICE_H__*/