/*
Copyright (c) 2010 Peter Barrett

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef SOCKET_H_INCLUDED
#define SOCKET_H_INCLUDED

#define SOCKET_HCI      1
#define SOCKET_L2CAP    2
#define SOCKET_RFCOM    3
#define SOCKET_SDP      4

#define SO_RECBUF   1
#define SO_SNDBUF   2
#define NOPROTOOPT  -100

typedef struct
{
    u8  AddressSpecific[0]; // BDADDR,psm etc
} SocketAddrHdr;

enum SocketState
{
    SocketState_Unknown,
    SocketState_Opening,
    SocketState_Open,
    SocketState_Closing,
    SocketState_Closed,
    SocketState_Listen,
    SocketState_Accepting,
    SocketState_L2CAP_WaitConnect = 8,
    SocketState_L2CAP_WaitConnectRsp,
    SocketState_L2CAP_WaitDisconnect,
    SocketState_L2CAP_Config_wait = 16,
    SocketState_L2CAP_Config_wait_send,
    SocketState_L2CAP_Config_wait_req,
    SocketState_L2CAP_Config_wait_rsp,
    SocketState_L2CAP_Config_wait_reqrsp  
};

typedef void (*SocketCallback)(int socket, SocketState state, const u8* data, int len, void* userData);

int Socket_Create(int type, SocketCallback callback, void* userData);   // Allocate a socket
int Socket_Open(int type, SocketAddrHdr* addr, SocketCallback callback, void* userData);   // Open a socket
int Socket_Listen(int type, int channel, SocketCallback callback, void* userData);   // Open a socket passive
int Socket_Accept(int type, int scid, int rxid, SocketCallback callback, void* userData);   // Open a socket for an incoming connection
int Socket_SetOpt(int socket, int so, int* data, int len);
int Socket_GetOpt(int socket, int so, int* data, int len);
int Socket_Send(int socket, const u8* data, int len);
int Socket_State(int socket);
int Socket_Close(int socket);

//===========================================================================
//===========================================================================
//  Don't need to look at or use anything below this line:
//  Internal representation of socket

class SocketHandler;
class SocketInternal
{
    public:

    u8 ID;
    u8 State;
    u8 Type;
    u8 B0;
    SocketCallback Callback;
    void* userData;
    u8  Data[0];    // Extra socket data starts here

    void Recv(const u8* data, int len)
    {
        Callback(ID,(SocketState)State,data,len,userData);
    }

    void SetState(SocketState state)
    {
        State = state;
        Callback(ID,(SocketState)State,0,0,userData);
        if (state == SocketState_Closed) {
          printf("Socket %d has been freed\n", ID);
          ID = 0;
        }
    }
};

class SocketHandler
{
    public:
    virtual int Create(SocketInternal* sock) { printf("SocketHandler::Create: not implemented for %s\n", Name()); return sock->ID;}
    virtual int Open(SocketInternal* sock, SocketAddrHdr* addr) = 0;
    virtual int Send(SocketInternal* sock, const u8* data, int len) = 0;
    virtual int Close(SocketInternal* sock) = 0;
    virtual int Listen(SocketInternal* sock, int channel) { printf("SocketHandler::Listen: not implemented for %s\n", Name());return 0;}    
    virtual int Accept(SocketInternal* sock, int scid, int rxid) { printf("SocketHandler::Accept: not implemented for %s\n", Name());return 0;}
    virtual int SetOpt(SocketInternal* sock, int so, int* data, int len) { printf("SocketHandler::SetOpt: not implemented for %s\n", Name());return 0;}
    virtual int GetOpt(SocketInternal* sock, int so, int* data, int len) { printf("SocketHandler::GetOpt: not implemented for %s\n", Name());return 0;}
    virtual char* Name() { return "Base_SocketHandler";}
};

int RegisterSocketHandler(int type, SocketHandler* handler);
SocketInternal* GetSocketInternal(int socket);

#define ERR_SOCKET_TYPE_NOT_FOUND -200
#define ERR_SOCKET_NOT_FOUND -201
#define ERR_SOCKET_NONE_LEFT -202
#define ERR_SOCKET_CANT_LISTEN -203

#endif // SOCKET_H_INCLUDED
