// Example

#include <string>

#include "mbed.h"
#include "eeprom.h"

#define EEPROM_ADDR 0x0   // I2c EEPROM address is 0x00

#define SDA p9            // I2C SDA pin
#define SCL p10           // I2C SCL pin

#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))

DigitalOut led2(LED2);

typedef struct _MyData {
                         int16_t sdata;
                         int32_t idata;
                         float fdata;
                       } MyData;

static void myerror(std::string msg)
{
  printf("Error %s\n",msg.c_str());
  exit(1);
}

void eeprom_test(void)
{
  EEPROM ep(SDA,SCL,EEPROM_ADDR,EEPROM::T24C64);  // 24C64 eeprom with sda = p9 and scl = p10
  uint8_t data[256],data_r[256];
  int8_t ival;
  uint16_t s;
  int16_t sdata,sdata_r;
  int32_t ldata[1024];
  int32_t eeprom_size,max_size;
  uint32_t addr;
  int32_t idata,idata_r;
  uint32_t i,j,k,l,t,id;
  float fdata,fdata_r;
  MyData md,md_r;
    
  eeprom_size = ep.getSize();
  max_size = MIN(eeprom_size,256);
  
  printf("Test EEPROM I2C model %s of %d bytes\n\n",ep.getName(),eeprom_size);
  
  // Test sequential read byte (max_size first bytes)
  for(i = 0;i < max_size;i++) {
     ep.read(i,ival);
     data_r[i] = ival;
     if(ep.getError() != 0)
       myerror(ep.getErrorMessage());
  }
  
  printf("Test sequential read %d first bytes :\n",max_size);
  for(i = 0;i < max_size/16;i++) {
     for(j = 0;j < 16;j++) {
        addr = i * 16 + j;
        printf("%3d ",(uint8_t)data_r[addr]);
     }
     printf("\n");
  }
    
    // Test sequential read byte (max_size last bytes)
  for(i = 0;i < max_size;i++) {
        addr = eeprom_size - max_size + i;
    ep.read(addr,ival);
    data_r[i] = ival;
    if(ep.getError() != 0)
      myerror(ep.getErrorMessage());
  }
  
  printf("\nTest sequential read %d last bytes :\n",max_size);
  for(i = 0;i < max_size/16;i++) {
     for(j = 0;j < 16;j++) {
        addr = i * 16 + j;
        printf("%3d ",(uint8_t)data_r[addr]);
     }
     printf("\n");
  }
  
  // Test write byte (max_size first bytes)
  for(i = 0;i < max_size;i++)
     data[i] = i;
  
  for(i = 0;i < max_size;i++) {
     ep.write(i,(int8_t)data[i]);
     if(ep.getError() != 0)
       myerror(ep.getErrorMessage());
  }
  
  // Test read byte (max_size first bytes)
  for(i = 0;i < max_size;i++) {
     ep.read(i,(int8_t&)ival);
     data_r[i] = (uint8_t)ival;
     if(ep.getError() != 0)
       myerror(ep.getErrorMessage());
  }
  
  printf("\nTest write and read %d first bytes :\n",max_size);
  for(i = 0;i < max_size/16;i++) {
     for(j = 0;j < 16;j++) {
        addr = i * 16 + j;
        printf("%3d ",(uint8_t)data_r[addr]);
     }
     printf("\n");
  }
  
  // Test current address read byte (max_size first bytes)
  ep.read((uint32_t)0,(int8_t&)ival); // current address is 0
  data_r[0] = (uint8_t)ival;
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  
  for(i = 1;i < max_size;i++) {
     ep.read((int8_t&)ival);
     data_r[i] = (uint8_t)ival;
     if(ep.getError() != 0)
       myerror(ep.getErrorMessage());
  }
  
  printf("\nTest current address read %d first bytes :\n",max_size);
  for(i = 0;i < max_size/16;i++) {
     for(j = 0;j < 16;j++) {
        addr = i * 16 + j;
        printf("%3d ",(uint8_t)data_r[addr]);
     }
     printf("\n");
  }
   
  // Test sequential read byte (first max_size bytes)
  ep.read((uint32_t)0,(int8_t *)data_r,(uint32_t) max_size);
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  
  printf("\nTest sequential read %d first bytes :\n",max_size);
  for(i = 0;i < max_size/16;i++) {
     for(j = 0;j < 16;j++) {
        addr = i * 16 + j;
        printf("%3d ",(uint8_t)data_r[addr]);
     }
     printf("\n");
  }
  
  // Test write short, long, float 
  sdata = -15202;
    addr = eeprom_size - 16;
  ep.write(addr,(int16_t)sdata); // short write at address eeprom_size - 16
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  
  idata = 45123;
    addr = eeprom_size - 12;
  ep.write(addr,(int32_t)idata); // long write at address eeprom_size - 12
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
    
  fdata = -12.26;
    addr = eeprom_size - 8;
  ep.write(addr,(float)fdata); // float write at address eeprom_size - 8
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  
  // Test read short, long, float
  printf("\nTest write and read short (%d), long (%d), float (%f) :\n",
           sdata,idata,fdata);  
  
  ep.read((uint32_t)(eeprom_size - 16),(int16_t&)sdata_r);
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  printf("sdata %d\n",sdata_r);
  
  ep.read((uint32_t)(eeprom_size - 12),(int32_t&)idata_r);
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  printf("idata %d\n",idata_r);
  
  ep.read((uint32_t)(eeprom_size - 8),fdata_r);
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  printf("fdata %f\n",fdata_r);
  
  // Test read and write a structure
  md.sdata = -15203;
  md.idata = 45124;
  md.fdata = -12.27;
 
  ep.write((uint32_t)(eeprom_size - 32),(void *)&md,sizeof(md)); // write a structure eeprom_size - 32
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
    
  printf("\nTest write and read a structure (%d %d %f) :\n",md.sdata,md.idata,md.fdata);
  
  ep.read((uint32_t)(eeprom_size - 32),(void *)&md_r,sizeof(md_r));
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
  
  printf("md.sdata %d\n",md_r.sdata);
  printf("md.idata %d\n",md_r.idata);
  printf("md.fdata %f\n",md_r.fdata);
    
    // Test read and write of an array of the first max_size bytes
    for(i = 0;i < max_size;i++)
       data[i] = max_size - i - 1;
    
    ep.write((uint32_t)(0),data,(uint32_t)max_size);
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
    
    ep.read((uint32_t)(0),data_r,(uint32_t)max_size);
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
    
    printf("\nTest write and read an array of the first %d bytes :\n",max_size);
    for(i = 0;i < max_size/16;i++) {
     for(j = 0;j < 16;j++) {
        addr = i * 16 + j;
        printf("%3d ",(uint8_t)data_r[addr]);
     }
     printf("\n");
  }
    printf("\n");
  
  // Test write and read an array of int32
  s = eeprom_size / 4;                // size of eeprom in int32
  int ldata_size = sizeof(ldata) / 4; // size of data array in int32
  l = s / ldata_size;                 // loop index
  
  // size of read / write in bytes
  t = eeprom_size;
  if(t > ldata_size * 4)
    t = ldata_size * 4;
  
  printf("Test write and read an array of %d int32 (write entire memory) :\n",t/4);

  // Write entire eeprom
    if(l) {
    for(k = 0;k < l;k++) {
       for(i = 0;i < ldata_size;i++)
          ldata[i] = ldata_size * k + i;
        
       addr = k * ldata_size * 4;
       ep.write(addr,(void *)ldata,t);
       if(ep.getError() != 0)
         myerror(ep.getErrorMessage());
    }  
    
      printf("Write OK\n");
    
    // Read entire eeprom
      id = 0;
    for(k = 0;k < l;k++) {
       addr = k * ldata_size * 4;
       ep.read(addr,(void *)ldata,t);
       if(ep.getError() != 0)
         myerror(ep.getErrorMessage());
  
       // format outputs with 8 words rows
       for(i = 0;i < ldata_size / 8;i++) {
                id++;
          printf("%4d ",id);
          for(j = 0;j < 8;j++) {
             addr = i * 8 + j;
             printf("%5d ",ldata[addr]);
          }
          printf("\n");
       }
    }
  }
    else {
        for(i = 0;i < s;i++)
       ldata[i] = i;
        
    addr = 0;
    ep.write(addr,(void *)ldata,t);
    if(ep.getError() != 0)
      myerror(ep.getErrorMessage());
        
        printf("Write OK\n");
    
    // Read entire eeprom
      id = 0;
    
    addr = 0;
    ep.read(addr,(void *)ldata,t);
    if(ep.getError() != 0)
      myerror(ep.getErrorMessage());
  
    // format outputs with 8 words rows
    for(i = 0;i < s / 8;i++) {
             id++;
       printf("%4d ",id);
       for(j = 0;j < 8;j++) {
          addr = i * 8 + j;
          printf("%5d ",ldata[addr]);
       }
       printf("\n");
    }
    }
  
  // clear eeprom
  printf("\nClear eeprom\n");

  ep.clear();
  if(ep.getError() != 0)
    myerror(ep.getErrorMessage());
    
  printf("End\n");  
    
}

int main() 
{

  eeprom_test();
    
  return(0);
}
