
/**
 * @file    SDShell.h
 * @brief   SD Card Utility - Emulate a basic UNIX terminal interface
 * @author  sam grove
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef SDSHELL_H
#define SDSHELL_H

#include "mbed.h"
#include "SDFileSystem.h"
#include "CommHandler.h"
#include "crc.h"

/** Example using the CommHandler class
 * @code
 *  #include "mbed.h"
 *  #include "SDFileSystem.h"
 *  #include "SDShell.h"
 *  
 *  Serial com(USBTX, USBRX);
 *  SDFileSystem sd(p11, p12, p13, p14, "sd");
 *  SDShell emulate;
 *
 *  int main()
 *  {
 *      emulate.init();
 *      emulate.shell(com, sd, "/sd");
 *  }
 * @endcode
 */

/**
 *  @class SDShell
 *  @brief API abstraction for a UNIX terminal enumator to the SDFileSystem
 */ 
class SDShell
{
public:
    
    /** Create the SDShell object
     */ 
    SDShell();
    
    /** Initialize members and compenents of the class
     */
    void init(void);
    
    /** Run the UNIX terminal emulator
     *  @param com - Address of an initialized Serial object
     *  @param storage - Address of an initialized SDFileSystem object
     *  @cwd - The name used to access the virtual filesystem
     */
    void shell(Serial &com, SDFileSystem &storage, char const *cwd);
    
private:
    Serial       *_com;
    SDFileSystem *_storage;
    CommHandler   _cmds;
    
    #define SHELL_BUF_SIZE  64
    #define SHELL_BUF_MASK  (SHELL_BUF_SIZE-1)
    char _cwd[SHELL_BUF_SIZE];
    char _buf[512];
    char _newpath[SHELL_BUF_SIZE];
    char _cmd[SHELL_BUF_SIZE];
    char _cmdline[SHELL_BUF_SIZE];
    char *_arg;
    uint32_t _debug;
    
    enum {UNKNOWN = 0, OK = '1', EXIT = '2'};
    
    // helper functions that build the interface
    void shellInput(void);
    char *split(char *dest, char *src, int max, char delim);
    void resolveDirectory(char *newpath, char *path);
    void splitName(char *path, char *dirname, char *basename);
    uint32_t match(char *arg1, char *arg2);
    
    // handler functions for each known input
    char *ls(char *cmd);
    char *cd(char *cmd);
    char *pwd(char *cmd);
    char *head(char *cmd);
    char *cat(char *cmd);
    char *bcat(char *cmd);
    char *touch(char *cmd);
    char *create(char *cmd);
    char *rm(char *cmd);
    char *exit(char *cmd);
    char *debug(char *cmd);
    char *du(char *cmd);
    char *cksum(char* cmd);
};

#endif
