/**
 * @file    S401M16KR.h
 * @brief   Device driver - Luminex S401M16KR 4 character 7 segment LCD
 * @author  sam grove & __________
 * @version 1.0
 * @see     http://www.lumex.com/specs/LCD-S401M16KR.pdf
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef S401M16KR_H
#define S401M16KR_H

#include "mbed.h"

/** Using the Luminex S401M16KR on a Freescale FRDM-KL46Z
 *
 * Example:
 * @code
 *  #include "mbed.h"
 *  #include "S401M16KR.h"
 *  
 *  S401M16KR lcd(PTD0, PTE4, PTB23, PTB22, PTC17, PTB21, PTB7, PTB8, PTE5, PTC18, PTB10, PTB11);
 *  
 *  int main()  
 *  {       
 *      int cnt = 0;
 *
 *      time_t rawtime = time(NULL);
 *      struct tm *timeinfo = localtime(&rawtime);
 *      lcd.init();
 *      lcd.enable();
 *
 *      while(1)
 *      {
 *        lcd.displayTime(12,45);
 *        wait(1.0f);
 *        lcd.displayTime(timeinfo);
 *        wait(1.0f);
 *        lcd.displayText("1234");
 *        wait(1.0f);
 *        lcd.displayTemp(26);
 *        wait(1.0f);
 *        lcd.printf("%s", asctime(timeinfo));
 *        wait(1.0f);
 *        lcd.printf("%d: Hello World\n", cnt++);
 *        wait(1.0f);
 *        lcd.puts("Goodbye");
 *        wait(1.0f);
 *      }
 *  }
 * @endcode
 */

/**
 *  @class S401M16KR
 *  @brief API for the Luminex S401M16KR 4 character 7 segment LCD
 */ 
class S401M16KR
{
private:


public:
    
    /** Create the S401M16KR object - pins not dynamically selectable
     */
    S401M16KR();
    
    /** Create the S401M16KR object
     *  @param com0 - Pin connected from LCD to MCU
     *  @param com1 - Pin connected from LCD to MCU
     *  @param com2 - Pin connected from LCD to MCU
     *  @param com3 - Pin connected from LCD to MCU
     *  @param _1d_1e_1g_1f - Pin connected from LCD to MCU
     *  @param dp1_1c_1b_1a - Pin connected from LCD to MCU
     *  @param _2d_2e_2g_2f - Pin connected from LCD to MCU
     *  @param dp2_2c_2b_2a - Pin connected from LCD to MCU
     *  @param _3d_3e_3g_3f - Pin connected from LCD to MCU
     *  @param dp3_3c_3b_3a - Pin connected from LCD to MCU
     *  @param _4d_4e_4g_4f - Pin connected from LCD to MCU     
     *  @param col_4c_4b_4a - Pin connected from LCD to MCU 
     */    
    S401M16KR(PinName com0, PinName com1, PinName com2, PinName com3
             ,PinName _1d_1e_1g_1f, PinName dp1_1c_1b_1a, PinName _2d_2e_2g_2f, PinName dp2_2c_2b_2a
             ,PinName _3d_3e_3g_3f, PinName dp3_3c_3b_3a, PinName _4d_4e_4g_4f, PinName col_4c_4b_4a);
    
    /** Clear state variables and initilize the dependant objects
     *  could be made private and called from constructor if not needed
     */
    void init(void) const;
    
    /** Turn on peripheral power and clocking 
     */
    void enable(void) const;
    
    /** Turn off peripheral power and clocking 
     */
    void disable(void) const;
    
    /** Scroll the display for text larger than 4 characters
     */
    void scrollSpeed(const float ms) const;
    
    /** Write the time to the display
     *  @param hours - time of day in hours
     *  @param minutes - time of day in minutes
     */
    void displayTime(const int hours, const int minutes) const;
    
    /** Write the time to the display
     *  @param time - the time of day as stored in struct tm
     */
    void displayTime(const struct tm time) const;
    
    /** Write text to the display
     *  @param str - A string to write
     *  @param scroll - If the string is too long scroll the contents if true
     */
    void displayText(const char *str, const int scroll = 0) const;
    
    /** Write temperature to the display
     *  @param temperature - a temperature value to display
     *  @param fahrenheit - If 0 the temperature is formatted as celcius (C) otherwise (T)
     */
    void displayTemp(const float temperature, const int fahrenheit = 0);
    
    /** Write a formatted string to the display
     *  @param format - The string + format specifiers
     *  @param ... - additional arguments
     *  @return - the amount of characters that will be displayed
     */
    int printf(const char *format, ...);
    
    /** Write a string to the display
     *  @param s - a string to be displayed
     *  @return - the amount of characters that will be displayed
     */
    int puts(const char *s);
};

#endif
