/**
 * @file    Oem615.h
 * @brief   Device driver - Novatel OEM615 GPS
 * @author  sam grove
 * @version 1.0
 * @see     http://www.novatel.com/support/firmware-software-and-manuals/product-manuals-and-doc-updates/oem6-family/
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef OEM615_H
#define OEM615_H
 
#include "LogUtil.h"
#include "mbed.h"

/** Using the Novatel OEM615
 *
 * Example:
 * @code
 *  #include "mbed.h"
 *
 *  int main()
 *  {
 *  }
 * @endcode
 */

/**
 *  @class Oem615
 *  @brief API abstraction for the OEM615 GPS
 */ 
class Oem615
{
public:

    Timer _from_pps;
    
    /** Create the Adis16488 object
     *  @param uart - A defined Serial object
     *  @param rst - A defined DigitalOut object
     *  @param pwr - A defined DigitalOut object
     *  @param pps - A defined InterruptIn object
     *  @param varf - A defined InterruptIn object
     */  
    Oem615(Serial &uart, DigitalOut &rst, DigitalOut &pwr, InterruptIn &pps, InterruptIn &varf);
    
    /** Clear state vars and initilize the dependant objects
     */
    void init(void);

    /** Allow the module to run and collect user input
     */
    void enable(void);
   
    /** Stop the module and put into low power mode
     */   
    void disable(void);
    
    // rx data storage
    #define GPS_RXBUF_SIZE (4096)
    #define GPS_RXBUF_MASK (GPS_RXBUF_SIZE-1)
    struct
    {
        volatile uint32_t loc;
        volatile uint8_t data[GPS_RXBUF_SIZE];
    }_rxbuf;
    
private:
    Serial      *_uart;     // tx, rx  //OEM615 GPS TX/RX for COM1 on GPS
    DigitalOut  *_rst;      // GPS Reset -- pull high for operation
    DigitalOut  *_pwr;      // reverse logic, 0 = on and 1 = off
    InterruptIn *_pps;      // replaced original pin p19 that doesnt have InterruptIn capabilites
    InterruptIn *_varf;     // VARF clock from GPS synched to 1PPS (100 Hz input as IMU command)
    
    void ppsHandler(void);
    void varfHandler(void);
    void uartRxHandler(void);
};
 
#endif
 
 
 