/**
 * @file    CommHandler.h
 * @brief   Core Utility - Manage ASCII communication between devices
 * @author  sam grove
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef COMMHANDLER_H
#define COMMHANDLER_H

#include "FP.h"
#include "LinkedList.h"
#include "LogUtil.h"

/** Example using the CommHandler class
 * @code
 *  #include "mbed.h"
 *  #include "CommHandler.h"
 *  
 *  DigitalOut myled(LED1);
 *  CommHandler msgs;
 *  
 *  char *one(char* msg)
 *  {
 *      // you can parse msg here
 *      LOG("\n");
 *      return msg;
 *  }
 *  
 *  class Wrap
 *  {
 *  public:
 *      Wrap(){}
 *      char *two(char *msg)
 *      {
 *          // you can parse msg here
 *          LOG("\n");
 *          return msg;
 *      }
 *  }obj;
 *  
 *  
 *  char *three(char* msg)
 *  {
 *      // you can parse msg here
 *      LOG("\n");
 *      return msg;
 *  }
 *  
 *  int main()
 *  {
 *      char *tmp = 0;
 *  
 *      msgs.attachMsg("One", &one);
 *      msgs.attachMsg("Two", &obj, &Wrap::two);
 *      msgs.attachMsg("Three", &three);
 *      
 *      tmp = msgs.messageLookup(0);
 *      printf("0:%s\n", tmp);
 *      tmp = msgs.messageLookup(1);
 *      printf("1:%s\n", tmp);
 *      tmp = msgs.messageLookup(2);
 *      printf("2:%s\n", tmp);
 *      tmp = msgs.messageLookup(3);
 *      printf("3:%s\n", tmp);
 *      tmp = msgs.messageLookup(4);
 *      printf("4:%s\n", tmp);
 *          
 *      tmp = msgs.serviceMessage("Two-00-66-99-20133");
 *      printf("1: Found: %s\n", tmp);
 *      tmp = msgs.serviceMessage("One-99-60-1-339788354");
 *      printf("2: Found: %s\n", tmp);
 *      tmp = msgs.serviceMessage("Three-xx-xx-XX-XXXXXXX");
 *      printf("3: Found: %s\n", tmp);
 *      
 *      error("End of Test\n");
 *  }
 * @endcode
 */

/**
 *  @class CommHandler
 *  @brief API abstraction for managing device to device communication
 */ 
class CommHandler
{
private:
    LinkedList <node>_list;

public:

    /**
     *  @struct MsgObject
     *  @brief An object to store in the linked list
     */
    struct MsgObj
    {
        char *string;               /*!< The header we are going to match */
        FP<char *, char *>handler;  /*!< The function to call when a match is found */
    };
    
    /** Create the CommHandler object
     */     
    CommHandler();
    
    /** Attach a member function as the match handler
     *  @param string - The string that we're trying to match
     *  @param item - Address of a initialized object
     *  @param member - Address of the initialized object's member function
     */
    template<class T>
    void attachMsg( char *string, T *item, char*(T::*method)(char *) )
    {
        MsgObj *new_node = new MsgObj [1];
        // make sure the new object was allocated
        if (NULL == new_node)
        {
            ERROR("Memory Allocation Failed\n");
        }
        // store the user parameters
        new_node->string = string;
        new_node->handler.attach( item, method );
        // and insert them into the list
        _list.append(new_node);
        return;
    }
    
    /** Attach a global function as the match handler
     *  @param string - The string that we're trying to match
     *  @param function - Address of the global function
     */
    void attachMsg( char *string, char *(*function)(char*) );
    
    /** called in a loop somewhere to processes messages
     *  @param buffer - A buffer containing ascii data
     *  @return Data from the handler message when a match is found and 0 otherwise
     */
    char *serviceMessage( char* buffer );
    
    
    /** Determine what a message in location X is looking to match
     *  @param loc - The location of the member in the list
     *  @return The message that is attached to the list
     */
    char *messageLookup( uint32_t const loc );

};

#endif

