/**
 * @file    BufferedSerial.cpp
 * @brief   Software Buffer - Extends mbed Serial functionallity adding irq driven TX and RX
 * @author  sam grove
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "BufferedSerial.h"
#include "LogUtil.h"
#include <stdarg.h>

BufferedSerial::BufferedSerial(PinName tx, PinName rx, const char* name)
    : Serial(tx, rx, name)
{
    Serial::attach(this, &BufferedSerial::rxIrq, Serial::RxIrq);
    Serial::attach(this, &BufferedSerial::txIrq, Serial::TxIrq);
    
    return;
}

BufferedSerial::~BufferedSerial(void)
{
    Serial::attach(NULL, Serial::RxIrq);
    Serial::attach(NULL, Serial::TxIrq);
    
    return;
}

int BufferedSerial::readable(void)
{
    return _rxbuf.available();  // note: look if things are in the buffer
}

int BufferedSerial::writeable(void)
{
    return 1;   // buffer allows overwriting by design, always true
}

int BufferedSerial::getc(void)
{
    return _rxbuf;   // note: pulling from the buffer
}

int BufferedSerial::putc(int c)
{
    _txbuf = (char)c;
    BufferedSerial::txIrq();    // prime the txirq to stoke transmit
    
    return c;
}

int BufferedSerial::puts(const char *s)
{
    const char *size = s;
    
    while(*(s) != 0)
    {
        _txbuf = *(s++);
    }
    BufferedSerial::txIrq();    // prime the txirq to stoke transmit
    
    return s - size;
}

int BufferedSerial::printf(const char* format, ...)
{
    char buf[256] = {0};
    int r = 0;
    
    va_list arg;
    va_start(arg, format);
    r = vsprintf(buf, format, arg);
    // this may not hit the heap but should alert the user anyways
    if(r > sizeof(buf))
    {
        ERROR("Buffer Overwrite Occured!\n");
    }
    r = BufferedSerial::puts(buf);
    va_end(arg);
    
    return r;
}

void BufferedSerial::rxIrq(void)
{
    // read from the peripheral and see if things are available
    while(serial_readable(&_serial))
    {
        _rxbuf = serial_getc(&_serial); // if so load them into a buffer
    }
    
    return;
}

void BufferedSerial::txIrq(void)
{
    // see if there is room in the hardware buffer and something is in the software buffer
    while(serial_writable(&_serial) && _txbuf.available())
    {
        serial_putc(&_serial, (int)_txbuf.get());
    }
    
    return;
}



