//**********************
// MCP3425.cpp for mbed
//
// (C)Copyright 2016 Satoshi Nihonyanagi, All Rights Reserved.
//
// (C)Copyright 2014 All rights reserved by Y.Onodera
// http://einstlab.web.fc2.com
//**********************

#include "mbed.h"
#include "MCP3425.h"

const int MCP3425::ADDR = 0xD0;
const float MCP3425::VREF = 2.048;

MCP3425::MCP3425(I2C& i2c)
    :  _i2c(i2c), _addr(ADDR)
{
    _init();
}

MCP3425::MCP3425(I2C& i2c, int addr)
    : _i2c(i2c), _addr(addr)
{
    _init();
}

void MCP3425::set(MCP3425::Gain gain)
{
    _cfg.bit.G = (unsigned char)gain;
    _write_cfg();
}

void MCP3425::set(MCP3425::Resolution resolution)
{
    _cfg.bit.S = (unsigned char)resolution;
    _write_cfg();
}

void MCP3425::set(MCP3425::Conversion conversion)
{
    _cfg.bit.OC = (unsigned char)conversion;
    _write_cfg();
}

short MCP3425::get()
{
    return _get_code();
}

float MCP3425::read()
{
    return _c2v(_get_code());
}

void MCP3425::_init()
{
    _cfg.bit.G = (unsigned char)G1;
    _cfg.bit.S = (unsigned char)W16;
    _cfg.bit.OC = (unsigned char)ONESHOT;
    _cfg.bit.C = 0;
    _cfg.bit.RDY = 0;

    _write_cfg();
}

void MCP3425::_write_cfg()
{
    _buf[0] = _cfg.UC;
    _i2c.write(_addr, _buf, 1);
}

short MCP3425::_get_code()
{
    // Trigger updating value if one-shot Mode
    if (_cfg.bit.OC == 0)
    {
        _cfg.bit.RDY = 1;
        _write_cfg();
    }

    while(1) {
        if( _i2c.read( ADDR, _buf, 3) != 0 )
        {
            // failed to read (nack)
            return 0;
        }

        _val.byte.HB = _buf[0];
        _val.byte.LB = _buf[1];
        _cfg.UC = _buf[2];
        
        if(_cfg.bit.RDY == 0 ) break;
        wait_ms(5); // fastest rate: 240 SPS with 12-bit
    }

    return _val.S;
}

float MCP3425::_c2v(short code)
{
    int gain = 1;
    if (_cfg.bit.G == 0) gain = 1;
    else if (_cfg.bit.G == 1) gain = 2;
    else if (_cfg.bit.G == 2) gain = 4;
    else if (_cfg.bit.G == 3) gain = 8;

    int maxcode = 2047;
    if (_cfg.bit.S == 0 ) maxcode = 2047;
    else if (_cfg.bit.S == 1 ) maxcode = 8191;
    else if (_cfg.bit.S == 2 ) maxcode = 32767;

    return code * VREF / gain / ( maxcode + 1 );    
}
