#include "mbed.h"
#include "SakuraIO.h"

#if defined(TARGET_LPC176X)
DigitalOut myled(LED1);
SakuraIO_I2C sakuraio(p28, p27); // sda, scl (mbed LPC1768)
/*
  or
I2C i2c(p28, p27); // sda, scl
SakuraIO_I2C sakuraio(i2c); // i2c
  or
SakuraIO_SPI sakuraio(p5, p6, p7, p8); // mosi, miso, sclk, cs
  or
SPI spi(p5, p6, p7); // mosi, miso, sclk
SakuraIO_SPI sakuraio(spi, p8); // spi, cs
*/
#elif defined(TARGET_KL25Z)
DigitalOut myled(LED2);
SakuraIO_I2C sakuraio(PTE0, PTE1); // sda, scl (FRDM KL25Z)
/*
  or
SakuraIO_SPI sakuraio(PTD2, PTD3, PTD1, PTD0); // mosi, miso, sclk, cs
*/
#endif

Serial pc(USBTX, USBRX);

int main () {
  int counter = 0;

  pc.baud(115200);
  pc.printf("Waiting to come online");
  for (;;) {
    if( (sakuraio.getConnectionStatus() & 0x80) == 0x80 ) break;
    pc.printf(".");
    myled = !myled;
    wait(1);
  }
  pc.printf("\r\n");
  myled = 1;

  for (;;) {
    counter ++;
    pc.printf("%d\r\n", counter);

    uint8_t request[33] = {};
    uint8_t response[33] = {};

    // Echoback
    request[0] = counter;
    sakuraio.echoback(1, request, response);
    pc.printf("%d -> %d\r\n", request[0], response[0]);

    // Unixtime
    uint32_t unixtime = (uint32_t)(sakuraio.getUnixtime() / 1000UL);
    pc.printf("Unixtime %d\r\n", unixtime);

    // ADC
    uint16_t adc;
    adc = sakuraio.getADC(0);
    pc.printf("ADC0 %d\r\n", adc);
    adc = sakuraio.getADC(1);
    pc.printf("ADC1 %d\r\n", adc);

    // ProductID
    uint16_t pid = sakuraio.getProductID();
    pc.printf("PID %d\r\n", pid);

    // UniqueID
    sakuraio.getUniqueID((char *)response);
    pc.printf("UID %s\r\n", (char *)response);

    // Version
    sakuraio.getFirmwareVersion((char *)response);
    pc.printf("Ver %s\r\n", (char *)response);

    // Connection Status
    uint8_t connectionStatus = sakuraio.getConnectionStatus();
    pc.printf("Status %d\r\n", connectionStatus);

    //getSignalQuarity
    uint8_t signalQuarity = sakuraio.getSignalQuarity();
    pc.printf("Quality %d\r\n", signalQuarity);


    // Tx Queue
    uint8_t ret;
    ret = sakuraio.enqueueTx((uint8_t)0, (int32_t)counter);
    pc.printf("Enqueue %d\r\n", ret);

    uint8_t avail;
    uint8_t queued;
    sakuraio.getTxQueueLength(&avail, &queued);
    pc.printf("Tx Available=%d ", avail);
    pc.printf("Queued=%d\r\n", queued);

    if (queued >= 30){
      ret = sakuraio.clearTx();
      pc.printf("Clear %d\r\n", ret);
    } else if (queued >= 5){
      ret = sakuraio.send();
      pc.printf("Send %d\r\n", ret);
    }

    // Rx Queue
    sakuraio.getRxQueueLength(&avail, &queued);
    pc.printf("Rx Available=%d ", avail);
    pc.printf("Queued=%d\r\n", queued);

    for (uint8_t i = 0; i < queued; i ++) {
      uint8_t channel;
      uint8_t type;
      uint8_t values[8];
      int64_t offset;
      ret = sakuraio.dequeueRx(&channel, &type, values, &offset);
      pc.printf("Dequeue %d", ret);
      if (ret == 0x01) {
        pc.printf(" ch=%d", channel);
        pc.printf(" type=%d", type);
        pc.printf(" values=[");
        for (uint8_t b = 0; b < 7; b ++) {
          pc.printf("%d ", values[b]);
        }
        pc.printf("%d] offset=%d\r\n", values[7], offset);
      } else {
        pc.printf(" ERROR\r\n");
      }
    }

    wait(1);
  }
}
