#include "Clock.h"
#include "Log.h"
#include "rtc_api.h"

extern Log _;

Clock* Clock::getInstance()
{
    static Clock* instance = new Clock();
    return instance;
}

//para conseguir un ss de frecuencia mayor a 1ms
//#define PREDIV_A 0x1F
//#define PREDIV_S 0x3FF

//valores por defecto
#define PREDIV_A (0x7FUL)
#define PREDIV_S (0xFFUL)


Clock::Clock()
{
    WakeUp::attach(HAL_PWR_DisableSleepOnExit);

    this->hibernate = false;
}

time_t Clock::getUnixTime()
{
    return time(NULL);
}

uint32_t Clock::getMilliSeconds()
{

    uint32_t intentos = 0;
    PWR->CR |= PWR_CR_DBP;
    while(((RTC->ISR & RTC_ISR_RSF) != 0) &&  intentos < 20){
        RTC->WPR = 0xCA;
        RTC->WPR = 0x53;

        RTC->ISR &=  ~RTC_ISR_RSF;

        intentos++;
    }
    if(intentos == 20){
        mbed_die();
    }
    intentos = 0;

    while(((RTC->ISR & RTC_ISR_RSF) == 0) &&  intentos < 2000) {
        intentos++;
    }
    if(intentos == 2000){
        mbed_die();
    }


    uint32_t ss = RTC->SSR;
    uint32_t tr = RTC->TR;
    uint32_t dr = RTC->DR;

    uint32_t milliseconds = ((tr>>12) & 0x7 ) * 600000  + ((tr>>8) & 0xF ) * 60000  + ((tr>>4) & 0x7) * 10000 + ((tr) & 0xF)*1000 + ((this->prediv_s - ss) * 1000) / (this->prediv_s + 1);
    return milliseconds;
}

const struct tm* Clock::getTmStruct()
{
    time_t seconds = time(NULL);
    return localtime(&seconds);
}

const string Clock::getTimeAsAString()
{
    time_t seconds = time(NULL);
    string str = ctime(&seconds);
    str.erase(str.size() - 1);// remove \n
    return str;
}

void Clock::updateUnixTime(time_t unixTime)
{
    set_time(unixTime);
}

void Clock::programAlarm(uint32_t seconds)
{
    // Clear Wakeup flag
    if (PWR->CSR & PWR_CSR_WUF) {
        PWR->CR |= PWR_CR_CWUF;
    }

    // Enable write access to RTC registers
    if (!(PWR->CR & PWR_CR_DBP)) {
        PWR->CR |= PWR_CR_DBP;
    }

    WakeUp::set(seconds);

    PWR->CR &= ~(PWR_CR_DBP);
}

void Clock::goToSleep() {
    uint32_t tmpreg = 0U;
    this->hibernate = true;
    /* Select the regulator state in Stop mode ---------------------------------*/
    tmpreg = PWR->CR;

    /* Clear PDDS and LPDS bits */
    CLEAR_BIT(tmpreg, (PWR_CR_PDDS | PWR_CR_LPSDSR));

    /* Set LPSDSR active to the converter run in low power mode */
    SET_BIT(tmpreg, PWR_CR_LPSDSR);

    /* Set ULP active ultra low power, this desative Vref regulated */
    SET_BIT(tmpreg, PWR_CR_ULP);


    /* Store the new value */
    PWR->CR = tmpreg;

    /* Set SLEEPDEEP bit of Cortex System Control Register */
    SET_BIT(SCB->SCR, SCB_SCR_SLEEPDEEP_Msk);

    /* Enable sleep on exit */
    HAL_PWR_EnableSleepOnExit();

    /* Request Wait For Interrupt */
    __WFI();

    /* Reset SLEEPDEEP bit of Cortex System Control Register */
    CLEAR_BIT(SCB->SCR, SCB_SCR_SLEEPDEEP_Msk);
}

void Clock::resetClockRegisters()
{
    RCC->CR |= (uint32_t)0x00000100;

    /*!< Reset SW[1:0], HPRE[3:0], PPRE1[2:0], PPRE2[2:0], MCOSEL[2:0] and MCOPRE[2:0] bits */
    RCC->CFGR &= (uint32_t) 0x88FF400C;

    /*!< Reset HSION, HSIDIVEN, HSEON, CSSON and PLLON bits */
    RCC->CR &= (uint32_t)0xFEF6FFF6;

    /*!< Reset HSI48ON  bit */
    RCC->CRRCR &= (uint32_t)0xFFFFFFFE;

    /*!< Reset HSEBYP bit */
    RCC->CR &= (uint32_t)0xFFFBFFFF;

    /*!< Reset PLLSRC, PLLMUL[3:0] and PLLDIV[1:0] bits */
    RCC->CFGR &= (uint32_t)0xFF02FFFF;

    /*!< Disable all interrupts */
    RCC->CIER = 0x00000000;
}

void Clock::initClock()
{
    resetClockRegisters();
    /* 1- If fail try to start with HSE and external xtal */
    if (SetSysClock_PLL_HSE(0) == 0)
    {
        /* 2- If fail start with HSI clock */
        if (SetSysClock_PLL_HSI() == 0)
        {
            mbed_die();
        }
    }

    if(!rtc_isenabled()) {
            rtc_init();
     }

    this->prediv_s = RTC->PRER & 0x00007FFF;

    uint32_t intentos = 0;
    PWR->CR |= PWR_CR_DBP;
    do {
        RTC->WPR = 0xCA;
        RTC->WPR = 0x53;
        RTC->ISR &=  ~RTC_ISR_RSF;
        intentos++;
    } while (((RTC->ISR & RTC_ISR_RSF) != 0) &&  intentos < 20);
    if(intentos == 20){
        mbed_die();
    }
    intentos = 0;
    while(((RTC->ISR & RTC_ISR_RSF) == 0) &&  intentos < 2000) {
        intentos++;
    }
    if(intentos == 2000){
        mbed_die();
    }

    this->hibernate = false;
}

uint8_t Clock::SetSysClock_PLL_HSE(uint8_t bypass)
{
    RCC_ClkInitTypeDef RCC_ClkInitStruct;
    RCC_OscInitTypeDef RCC_OscInitStruct;

    /* Used to gain time after DeepSleep in case HSI is used */
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_HSIRDY) != RESET) {
        return 0;
    }

    /* The voltage scaling allows optimizing the power consumption when the device is
       clocked below the maximum system frequency, to update the voltage scaling value
       regarding system frequency refer to product datasheet. */
    __PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

    /* Enable HSE and  activate PLL with HSE as source */
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
    if (bypass == 0) {
        RCC_OscInitStruct.HSEState          = RCC_HSE_ON; /* External 8 MHz xtal on OSC_IN/OSC_OUT */
    } else {
        RCC_OscInitStruct.HSEState          = RCC_HSE_BYPASS; /* External 8 MHz clock on OSC_IN */
    }
    RCC_OscInitStruct.HSIState            = RCC_HSI_OFF;
    RCC_OscInitStruct.HSI48State          = RCC_HSI48_OFF; /* For USB and RNG clock */
    // PLLCLK = (8 MHz * 8)/2 = 32 MHz
    RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSE;
    RCC_OscInitStruct.PLL.PLLMUL          = RCC_PLLMUL_8;
    RCC_OscInitStruct.PLL.PLLDIV          = RCC_PLLDIV_2;
    if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK) {
        return 0; // FAIL
    }

    /* Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2 clocks dividers */
    RCC_ClkInitStruct.ClockType      = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
    RCC_ClkInitStruct.SYSCLKSource   = RCC_SYSCLKSOURCE_PLLCLK; // 32 MHz
    RCC_ClkInitStruct.AHBCLKDivider  = RCC_SYSCLK_DIV1;         // 32 MHz
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;           // 32 MHz
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;           // 32 MHz
    if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_1) != HAL_OK) {
        return 0; // FAIL
    }

    return 1; // OK
}

uint8_t Clock::SetSysClock_PLL_HSI(void)
{
    RCC_ClkInitTypeDef RCC_ClkInitStruct;
    RCC_OscInitTypeDef RCC_OscInitStruct;

    /* The voltage scaling allows optimizing the power consumption when the device is
       clocked below the maximum system frequency, to update the voltage scaling value
       regarding system frequency refer to product datasheet. */
    __PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

    /* Enable HSI oscillators and activate PLL with HSI as source */
    RCC_OscInitStruct.OscillatorType      = RCC_OSCILLATORTYPE_HSI;
    RCC_OscInitStruct.HSEState            = RCC_HSE_OFF;
    RCC_OscInitStruct.HSIState            = RCC_HSI_ON;
    RCC_OscInitStruct.HSICalibrationValue = 16;
    RCC_OscInitStruct.HSI48State          = RCC_HSI48_ON; /* For USB and RNG clock */
    // PLLCLK = (16 MHz * 4)/2 = 32 MHz
    RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSI;
    RCC_OscInitStruct.PLL.PLLMUL          = RCC_PLLMUL_4;
    RCC_OscInitStruct.PLL.PLLDIV          = RCC_PLLDIV_2;
    if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK) {
        return 0; // FAIL
    }

    /* Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2 clocks dividers */
    RCC_ClkInitStruct.ClockType      = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
    RCC_ClkInitStruct.SYSCLKSource   = RCC_SYSCLKSOURCE_PLLCLK; // 32 MHz
    RCC_ClkInitStruct.AHBCLKDivider  = RCC_SYSCLK_DIV1;         // 32 MHz
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;           // 32 MHz
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;           // 32 MHz
    if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_1) != HAL_OK) {
        return 0; // FAIL
    }
    return 1; // OK
}

uint8_t Clock::SetSysClock_MSI(){

    uint8_t intentos = 0;

   //asegurarme que MSI este desabilitado

   if((RCC->CR & RCC_CR_MSION) != 0) {
       //esta habilitado
       RCC->CR &= ~RCC_CR_MSION;
       while((RCC->CR & RCC_CR_MSIRDY) != 0 && intentos < 50){
                intentos++;
       }

       if (intentos == 50) {
              return 0;
       }
   }

   //configurar frecuencia MSI 2.097MHz
   RCC->ICSCR = (RCC->ICSCR & (~RCC_ICSCR_MSIRANGE_Msk)) | RCC_ICSCR_MSIRANGE_5;


   //habilitar MSI y esperar que este listo
   RCC->CR |= RCC_CR_MSION;

   intentos = 0;
   while((RCC->CR & RCC_CR_MSIRDY) == 0 && intentos < 50){
       intentos++;
   }

   if (intentos == 50) {
       return 0;
   }

   // configurar como fuente de reloj del sistema MSI y configurara como reloj de despertar el MSI

   RCC->CFGR = RCC_CFGR_MCOPRE_DIV1 | RCC_CFGR_MCOSEL_NOCLOCK | RCC_CFGR_PLLDIV2 | RCC_CFGR_PLLMUL3
           | RCC_CFGR_PLLSRC_HSE /*| RCC_CFGR_STOPWUCK_Msk*/ | RCC_CFGR_PPRE2_DIV1 | RCC_CFGR_PPRE1_DIV1 | RCC_CFGR_HPRE_DIV1 | RCC_CFGR_SW_MSI;


   // comprobar que la fuente de reloj cambia a MSI
   intentos = 0;
  while((RCC->CFGR & RCC_CFGR_SWS_Msk) != RCC_CFGR_SW_MSI && intentos < 50){
      intentos++;
  }
  if (intentos == 50) {
         return 0;
  }

  //desactivar el resto de fuentes de reloj.
  RCC->CR &= RCC_CR_MSION | RCC_CR_MSIRDY; // desactivar PLL HSE HSI16
  RCC->CRRCR = 0; // desactivar HSI48

  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE3); // voltage select 1.2V

  return 1; //OK

}


void Clock::printClockInfo()
{
    switch (__HAL_RCC_GET_SYSCLK_SOURCE()) {
        case RCC_SYSCLKSOURCE_STATUS_MSI:
            _.print("MSI used as system clock.").ln();
            break;
        case RCC_SYSCLKSOURCE_STATUS_HSI:
            _.print("HSI used as system clock.").ln();
            break;
        case RCC_SYSCLKSOURCE_STATUS_HSE:
            _.print("HSE used as system clock.").ln();
            break;
        case RCC_SYSCLKSOURCE_STATUS_PLLCLK:
            _.print("PLL used as system clock.").ln();
            if ((RCC->CFGR & RCC_CFGR_PLLSRC) == RCC_CFGR_PLLSRC_HSE) {
                _.print("HSE oscillator clock selected as PLL input clock.").ln();
            } else {
                _.print("HSI16 oscillator clock selected as PLL input clock.").ln();
            }
            break;
    }
    HAL_RCC_GetHCLKFreq();
    _.print("CPU SystemCoreClock is ").print(SystemCoreClock).print(" Hz").ln();
}

#define SCB_SIZE 9
#define RCC_SIZE 21
#define PWR_SIZE 2
#define RTC_SIZE 25
#define SYSCFG_SIZE 9
#define TOTAL_SIZE SCB_SIZE + RCC_SIZE + PWR_SIZE + RTC_SIZE + SYSCFG_SIZE

void Clock::readRegisters(){
    //SCB registers

    _.print("SCB\r\n");
    _.print("Offset\tValue\r\n");
    uint32_t pto = SCB_BASE;
    for(uint32_t i = 0 ; i < SCB_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,*(uint32_t *)(pto+i*4));
    }

    //RCC registers

    _.print("RCC\r\n");
    _.print("Offset\tValue\r\n");
    pto = RCC_BASE;
    for(uint32_t i = 0 ; i < RCC_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,*(uint32_t *)(pto+i*4));
    }

    //PWR registers
    _.print("PWR\r\n");
    _.print("Offset\tValue\r\n");
    pto = PWR_BASE;
    for(uint32_t i = 0 ; i < PWR_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,*(uint32_t *)(pto+i*4));
    }

    //RTC registers
    _.print("RTC\r\n");
    _.print("Offset\tValue\r\n");
    pto = RTC_BASE;
    for(uint32_t i = 0 ; i < RTC_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,*(uint32_t *)(pto+i*4));
    }

    //SYSCFG registers
    _.print("SYSCFG\r\n");
    _.print("Offset\tValue\r\n");
    pto = SYSCFG_BASE;
    for(uint32_t i = 0 ; i < SYSCFG_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,*(uint32_t *)(pto+i*4));
    }
    _.ln();
    _.flush();
}


/*void Clock::printRegisters(){
    bool deleteReg = false;
    if(this->registers!=NULL) {
        deleteReg = true;
    } else {
        this->registers = (uint32_t*)(Eeprom::getInstance()->readLongInt(Address::Register::BASE));
    }
    _.print("SCB\r\n");
    _.print("Offset\tValue\r\n");

    uint8_t p = 0;

    for(uint32_t i = 0 ; i < SCB_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,this->registers[p]);
        p++;
        if(p >= TOTAL_SIZE){
            return;
        }
    }

    _.print("RCC\r\n");
    _.print("Offset\tValue\r\n");

    for(uint32_t i = 0 ; i < RCC_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,this->registers[p]);
        p++;
        if(p >= TOTAL_SIZE){
            return;
        }
    }

    _.print("PWR\r\n");
    _.print("Offset\tValue\r\n");

    for(uint32_t i = 0 ; i < PWR_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,this->registers[p]);
        p++;
        if(p >= TOTAL_SIZE){
            return;
        }
    }

    _.print("RTC\r\n");
    _.print("Offset\tValue\r\n");

    for(uint32_t i = 0 ; i < RTC_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,this->registers[p]);
        p++;
        if(p >= TOTAL_SIZE){
            return;
        }
    }

    _.print("SYSCFG\r\n");
    _.print("Offset\tValue\r\n");

    for(uint32_t i = 0 ; i < SYSCFG_SIZE ; i++) {
        _.printf("0x%02X\t0x%08X\r\n",i*4,this->registers[p]);
        p++;
        if(p >= TOTAL_SIZE){
            return;
        }
    }
    if(deleteReg == true) {
        delete[] this->registers;
    }
    this->registers = NULL;



}*/