// ----------------------------------------------------------------------------
// Copyright 2016-2018 ARM Ltd.
//
// SPDX-License-Identifier: Apache-2.0
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ----------------------------------------------------------------------------
#ifndef MBED_TEST_MODE

#include "mbed.h"
#include "simple-mbed-cloud-client.h"
#include "LittleFileSystem.h"

#include "stdbool.h" //for stepper motor function

#define STEPPER_PIN_1 PA_15  //ARD.D9-PWM PA15
#define STEPPER_PIN_2 PA_2 //ARD.D10-PWM PA2
#define STEPPER_PIN_3 PA_7 //ARD.D11-PWM PA7
#define STEPPER_PIN_4 PA_6 //ARD.D12-PWM PA6
 
DigitalOut steeperOne(STEPPER_PIN_1);
DigitalOut steeperTwo(STEPPER_PIN_2);
DigitalOut steeperThree(STEPPER_PIN_3);
DigitalOut steeperFour(STEPPER_PIN_4);
 
DigitalOut myIOs(PB_14); // review the results on PB_14
DigitalOut myIOs2(PA_5); // review the results on PA_5
int step_number=0;
int open_flag=0;
int close_flag=0;

// Default network interface object. Don't forget to change the WiFi SSID/password in mbed_app.json if you're using WiFi.
NetworkInterface *net;

// Default block device available on the target board
BlockDevice* bd = BlockDevice::get_default_instance();
SlicingBlockDevice sd(bd, 0, 2*1024*1024);

#if COMPONENT_SD || COMPONENT_NUSD
// Use FATFileSystem for SD card type blockdevices
FATFileSystem fs("fs");
#else
// Use LittleFileSystem for non-SD block devices to enable wear leveling and other functions
LittleFileSystem fs("fs");
#endif

// Default User button for GET example and for resetting the storage
InterruptIn button(BUTTON1);
// Default LED to use for PUT/POST example
DigitalOut led(LED1, 1);

// How often to fetch sensor data (in seconds)
#define SENSORS_POLL_INTERVAL 3.0

// Send all sensor data or just limited (useful for when running out of memory)
#define SEND_ALL_SENSORS

// Sensors related includes and initialization
#include "HTS221Sensor.h" //humidity and temperature
#include "LPS22HBSensor.h"  //Barometer

static DevI2C devI2c(PB_11,PB_10);
static HTS221Sensor sen_hum_temp(&devI2c);
static LPS22HBSensor sen_press_temp(&devI2c);
static DigitalOut shutdown_pin(PC_6);
//static VL53L0X sen_distance(&devI2c, &shutdown_pin, PC_7);

// Temperature reading from microcontroller
AnalogIn adc_temp(ADC_TEMP);
// Voltage reference reading from microcontroller
AnalogIn adc_vref(ADC_VREF);

// Declaring pointers for access to Pelion Client resources outside of main()
MbedCloudClientResource *res_button;
MbedCloudClientResource *res_led;

// Additional resources for sensor readings
#ifdef SEND_ALL_SENSORS
MbedCloudClientResource *res_humidity;
MbedCloudClientResource *res_pressure;
MbedCloudClientResource *res_adc_temp;
#endif /* SEND_ALL_SENSORS */

// An event queue is a very useful structure to debounce information between contexts (e.g. ISR and normal threads)
// This is great because things such as network operations are illegal in ISR, so updating a resource in a button's fall() function is not allowed
EventQueue eventQueue;

// When the device is registered, this variable will be used to access various useful information, like device ID etc.
static const ConnectorClientEndpointInfo* endpointInfo;

/**
 * PUT handler
 * @param resource The resource that triggered the callback
 * @param newValue Updated value for the resource
 */
void put_callback(MbedCloudClientResource *resource, m2m::String newValue) {
    printf("*** PUT received, new value: %s                             \n", newValue.c_str());
    led = atoi(newValue.c_str());
}

/**
 * POST handler
 * @param resource The resource that triggered the callback
 * @param buffer If a body was passed to the POST function, this contains the data.
 *               Note that the buffer is deallocated after leaving this function, so copy it if you need it longer.
 * @param size Size of the body
 */
void post_callback(MbedCloudClientResource *resource, const uint8_t *buffer, uint16_t size) {
    printf("*** POST received (length %u). Payload: ", size);
    for (size_t ix = 0; ix < size; ix++) {
        printf("%02x ", buffer[ix]);
    }
    printf("\n");
}
//Serial port define for uart1 usb
//pb7 rx
//pb6 tx
//Serial device(USBTX, USBRX);
Serial device(PB_6, PB_7);  // tx, rx


/**
 * Button function triggered by the physical button press.
 */
 
 
 
 
 void Tx_interrupt();
void Rx_interrupt();
void send_line();
void read_line();
 
 
// Circular buffers for serial TX and RX data - used by interrupt routines
const int buffer_size = 255;
// might need to increase buffer size for high baud rates
char tx_buffer[buffer_size+1];
char rx_buffer[buffer_size+1];
// Circular buffer pointers
// volatile makes read-modify-write atomic 
volatile int tx_in=0;
volatile int tx_out=0;
volatile int rx_in=0;
volatile int rx_out=0;
// Line buffers for sprintf and sscanf
char tx_line[80];
char rx_line[80];

     int i=0;
    int rx_i=0;
    device.baud(115200);
 
// Setup a serial interrupt function to receive data
    device.attach(&Rx_interrupt, Serial::RxIrq);
// Setup a serial interrupt function to transmit data
    device.attach(&Tx_interrupt, Serial::TxIrq);
 
 
void button_press() {
    int v = res_button->get_value_int() + 1;
    res_button->set_value(v);
    printf("*** Button clicked %d times                                 \n", v);
}
 
// Called everytime a new character goes into
// the RX buffer. Test that character for \n
// Note, rxGetLastChar() gets the last char that
// we received but it does NOT remove it from
// the RX buffer.
void read_line() {
    int i;
    i = 0;
// Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(UART1_IRQn);
// Loop reading rx buffer characters until end of line character
    while ((i==0) || (rx_line[i-1] != '\r')) {
// Wait if buffer empty
        if (rx_in == rx_out) {
// End Critical Section - need to allow rx interrupt to get new characters for buffer
            NVIC_EnableIRQ(UART1_IRQn);
            while (rx_in == rx_out) {
            }
// Start Critical Section - don't interrupt while changing global buffer variables
            NVIC_DisableIRQ(UART1_IRQn);
        }
        rx_line[i] = rx_buffer[rx_out];
        i++;
        rx_out = (rx_out + 1) % buffer_size;
    }
// End Critical Section
    NVIC_EnableIRQ(UART1_IRQn);
    rx_line[i-1] = 0;
    return;
}
/**
 * Notification callback handler
 * @param resource The resource that triggered the callback
 * @param status The delivery status of the notification
 */
void button_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status) {
    printf("*** Button notification, status %s (%d)                     \n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

/**
 * Registration callback handler
 * @param endpoint Information about the registered endpoint such as the name (so you can find it back in portal)
 */
void registered(const ConnectorClientEndpointInfo *endpoint) {
    printf("Registered to Pelion Device Management. Endpoint Name: %s\n", endpoint->internal_endpoint_name.c_str());
    endpointInfo = endpoint;
}

/*
    for one rotation when passed true or false in function OneStep(bool)
*/
void OneStep(bool dir){
if(dir)
{
    switch(step_number){
      case 0:
          steeperOne =1;
          steeperTwo =0;
          steeperThree =0;
          steeperFour =0;
        break;
      case 1:
          steeperOne =0;
          steeperTwo =1;
          steeperThree =0;
          steeperFour =0;
        break;
      case 2:
          steeperOne =0;
          steeperTwo =0;
          steeperThree =1;
          steeperFour =0;
        break;
      case 3:
          steeperOne =0;
          steeperTwo =0;
          steeperThree =0;
          steeperFour =1;
         break;
        }// end of switch 
}// end of if
  
else
{
    switch(step_number){
      case 0:
          steeperOne =0;
          steeperTwo =0;
          steeperThree =0;
          steeperFour =1;
        break;
      case 1:
          steeperOne =0;
          steeperTwo =0;
          steeperThree =1;
          steeperFour =0;
        break;
      case 2:
          steeperOne =0;
          steeperTwo =1;
          steeperThree =0;
          steeperFour =0;
        break;
      case 3:
          steeperOne =1;
          steeperTwo =0;
          steeperThree =0;
          steeperFour =0;
        break;
 
  
    }// end of switch
  }// end of else
step_number++;
  if(step_number > 3){
    step_number = 0;
  }
} // end of OneStep() function
/**
 * Initialize sensors
 */
void sensors_init() {
    uint8_t id1, id2;

    printf ("\nSensors configuration:\n");
    // Initialize sensors
    sen_hum_temp.init(NULL);
    sen_press_temp.init(NULL);

    /// Call sensors enable routines
    sen_hum_temp.enable();
    sen_press_temp.enable();

    sen_hum_temp.read_id(&id1);
    sen_press_temp.read_id(&id2);

    printf("HTS221  humidity & temperature    = 0x%X\n", id1);
    printf("LPS22HB pressure & temperature    = 0x%X\n", id2);

    printf("\n"); ;
}

/**
 * Update sensors and report their values.
 * This function is called periodically.
 */
void sensors_update() {
    float temp3_value, humid_value, pressure_value = 0.0;

    sen_hum_temp.get_humidity(&humid_value);
    sen_press_temp.get_pressure(&pressure_value);
    temp3_value = adc_temp.read()*100;
    
    printf("%2.0f^%2.0f^%3.0f",temp3_value,humid_value,pressure_value);   
    printf("\n");
           
   // printf ("%02d : %02d : %02d\n", temp3_value, hummitly, pressure);
    
// added a new code for stepper motor 
// if temperature reading is higher, open motor
    if(temp3_value > 30)
    {
       open_flag = 0;
       while(open_flag==0){
            OneStep(false);
            wait(0.01);
            bool btn_pressed = (button.read() == MBED_CONF_APP_BUTTON_PRESSED_STATE);
            if (btn_pressed) {
                printf("Door Opened...\n");
                open_flag=1;
                close_flag=0;
            }//end of if(btn_pressed)
     }// end of while
    }// end of if(temp3_value>20)
    // now we expect to close
    
    if(temp3_value < 20)//door has to close
    {
        close_flag = 0;
        while(close_flag==0){
            OneStep(true);
            wait(0.01);
            bool btn_pressed = (button.read() == MBED_CONF_APP_BUTTON_PRESSED_STATE);
            if (btn_pressed) {
                printf("Door Closed...\n");
                open_flag=0;
                close_flag=1;
            }//end of if(btn_pressed)
     }// end of while     
    }

    if (endpointInfo) {
#ifdef SEND_ALL_SENSORS
        res_humidity->set_value(humid_value);
        res_pressure->set_value(pressure_value);
        res_adc_temp->set_value(temp3_value);
#endif /* SEND_ALL_SENSORS */
    }
}

int main(void) 
{
    printf("\nStarting Simple Pelion Device Management Client example\n");

    int storage_status = fs.mount(&sd);
    if (storage_status != 0) {
        printf("Storage mounting failed.\n");
    }
    // If the User button is pressed ons start, then format storage.
    bool btn_pressed = (button.read() == MBED_CONF_APP_BUTTON_PRESSED_STATE);
    if (btn_pressed) {
        printf("User button is pushed on start...\n");
    }

    if (storage_status || btn_pressed) {
        printf("Formatting the storage...\n");
        int storage_status = StorageHelper::format(&fs, &sd);
        if (storage_status != 0) {
            printf("ERROR: Failed to reformat the storage (%d).\n", storage_status);
        }
    } else {
        printf("You can hold the user button during boot to format the storage and change the device identity.\n");
    }

    sensors_init();

    // Connect to the internet (DHCP is expected to be on)
    printf("Connecting to the network using Wifi...\n");
    net = NetworkInterface::get_default_instance();

    nsapi_error_t net_status = -1;
    for (int tries = 0; tries < 3; tries++) {
        net_status = net->connect();
        if (net_status == NSAPI_ERROR_OK) {
            break;
        } else {
            printf("Unable to connect to network. Retrying...\n");
        }
    }

    if (net_status != NSAPI_ERROR_OK) {
        printf("ERROR: Connecting to the network failed (%d)!\n", net_status);
        return -1;
    }

    printf("Connected to the network successfully. IP address: %s\n", net->get_ip_address());

    printf("Initializing Pelion Device Management Client...\n");

    // SimpleMbedCloudClient handles registering over LwM2M to Pelion DM
    SimpleMbedCloudClient client(net, bd, &fs);
    int client_status = client.init();
    if (client_status != 0) {
        printf("ERROR: Pelion Client initialization failed (%d)\n", client_status);
        return -1;
    }

    // Creating resources, which can be written or read from the cloud
    res_button = client.create_resource("3200/0/5501", "Button Count");
    res_button->set_value(0);
    res_button->methods(M2MMethod::GET);
    res_button->observable(true);
    res_button->attach_notification_callback(button_callback);

    res_led = client.create_resource("3201/0/5853", "LED State");
    res_led->set_value(1);
    res_led->methods(M2MMethod::GET | M2MMethod::PUT);
    res_led->attach_put_callback(put_callback);

#ifdef SEND_ALL_SENSORS
    // Sensor resources
/*    res_temperature = client.create_resource("3303/0/5700", "Temperature HTS221 (C)");
    res_temperature->set_value(0);
    res_temperature->methods(M2MMethod::GET);
    res_temperature->observable(true);*/

    res_humidity = client.create_resource("3304/0/5700", "Humidity");
    res_humidity->set_value(0);
    res_humidity->methods(M2MMethod::GET);
    res_humidity->observable(true);

    res_adc_temp = client.create_resource("3303/2/5700", "Temperature ADC (C)");
    res_adc_temp->set_value(0);
    res_adc_temp->methods(M2MMethod::GET);
    res_adc_temp->observable(true);

    res_pressure = client.create_resource("3323/0/5700", "Pressure");
    res_pressure->set_value(0);
    res_pressure->methods(M2MMethod::GET);
    res_pressure->observable(true);

#endif /* SEND_ALL_SENSORS */

    printf("Initialized Pelion Client. Registering...\n");

    // Callback that fires when registering is complete
    client.on_registered(&registered);

    // Register with Pelion DM
    client.register_and_connect();

    int i = 600; // wait up 60 seconds before attaching sensors and button events
    while (i-- > 0 && !client.is_client_registered()) {
        wait_ms(100);
    }

    button.fall(eventQueue.event(&button_press));

    // The timer fires on an interrupt context, but debounces it to the eventqueue, so it's safe to do network operations
    Ticker timer;
    timer.attach(eventQueue.event(&sensors_update), SENSORS_POLL_INTERVAL);

    // You can easily run the eventQueue in a separate thread if required
    eventQueue.dispatch_forever();
}

#endif
