/* Copyright (c) 2014 Shigenori Inoue, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
#ifndef __SCP1000__
#define __SCP1000__
#include "mbed.h"

/**  Example:
 * @code
 * #include "mbed.h"
 * #include "SCP1000.h"
 *
 * SCP1000 scp(p5, p6, p7, p18, p21, p19);
 *
 * int main()
 * {
 *     // Variables
 *     float temp, press;
 *
 *     // SCP1000 Initialization
 *     err = scp.init();
 *     if (err == SCP1000::ERR) {
 *         printf("Error!");
 *     return 1;
 *     }
 *
 *     while(true) {
 *         do {} while (scp.IsReady() == false);
 *         temp = scp.readTemperature();
 *         press = scp.readPressure();
 *         printf("Temperature: %2.2f degC\r\n", temp);
 *         printf("Atmospheric Pressure: %2.2f hPa", press);
 *     }
 * }
 * @endcode
 */

class SCP1000
{
public:

    /** Create an SCP1000 interface
    * @param mosi Master-Output-Slave-Input line of SPI
    * @param miso Master-Input-Slave-Output line of SPI
    * @param sclk Serial Clock of SPI
    * @param cs Chip Select signal to SCP1000
    * @param trig Measuremnt Trigger (TRIG) to SCP1000
    * @param drdy Data Ready (DRDY) signal from SCP1000
    */
    SCP1000(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName trig, PinName drdy);
    ~SCP1000();

    /** Initialize the SCP1000 */
    int init(void);

    /** Set the measurement mode of the SCP1000
     * @param mode Measurement mode:
     *   0-HIGH_RESOLUTION,
     *   1-HIGH_SPEED,
     *   2-ULTRA_LOW_POWER,
     *   3-LOW_POWER_17BIT,
     *   4-LOW_POWER_15BIT
     */
    void setMode(int mode);

    /** Trigger the measurement by TRIG wire in Low Power Mode
     * NOTE: IT CURRENTLY SEEMS NOT WORKING.
     * PLEASE USE triggerSoft() INSTEAD.
     */
    void triggerHard(void);

    /** Trigger the measurement by register writing in Low Power Mode */
    void triggerSoft(void);

    /** Check the data readiness
     * @return DRDY pin status: false-Not Ready, true-Ready
     */
    bool IsReady(void);

    /** Read temperature measurement result in degrees Celcius
     * @return Temperature in degrees Celcius
     */
    float readTemperature(void);

    /** Read atmospheric pressure measurement result in hectopascal
     * @return Atmospheric pressure in hPa
     */
    float readPressure(void);

    enum SCP1000_MODE {
        HIGH_RESOLUTION = 0,
        HIGH_SPEED = 1,
        ULTRA_LOW_POWER = 2,
        LOW_POWER_17BIT = 3,
        LOW_POWER_15BIT = 4
    };

    enum SCP1000_ERROR {
        OK = 0,
        ERR = 1
    };

private:
    SPI _s;
    DigitalOut _cs;
    DigitalOut _trig;
    DigitalIn _drdy;
    void write(int addr, int data);
    int read8(int addr);
    int read16(int addr);

    enum SCP1000_ADDRESS {
        REVID = 0x00,
        DATAWR = 0x01,
        ADDPTR = 0x02,
        OPERATIONS = 0x03,
        OPSTATUS = 0x04,
        RSTR = 0x06,
        STATUS = 0x07,
        DATARD8 = 0x1F,
        DATARD16 = 0x20,
        TEMPOUT = 0x21,
        CFG = 0x00,
        TWIADD = 0x05,
        USERDATA1 = 0x29,
        USERDATA2 = 0x2A,
        USERDATA3 = 0x2B,
        USERDATA4 = 0x2C
    };
};

#endif