/*****************************************************************************
**                                                                          **
**                      BIOROBOTICS 2016 GROUP 8                            **
**                      GROUP MEMBERS                                       **
**                                                                          **
**                      ANNELIES DE JONG      s1609580                      **
**                      INGMAR LOOHUIS        s1607359                      **
**                      ROBBERT VAN MIDDELAAR sjdjdjdj                      **
**                      MARGREET MORSINK      s1558382                      **
**                      KARLIJN NIJHUIS       s1777777                      **
**                                                                          **
******************************************************************************/

#include "mbed.h"
#include "BiQuad.h"
#include "MODSERIAL.h"
#include "QEI.h"
#include "math.h"

//=========== PORTS ==========================================================
AnalogIn emg1(A0); //right biceps
AnalogIn emg2(A1); //left biceps
Serial pc(USBTX,USBRX);

PwmOut motor1MagnitudePin(D5);
PwmOut motor2MagnitudePin(D6);
DigitalOut motor1DirectionPin (D4);
DigitalOut motor2DirectionPin (D7);
 
QEI encoder_m1(D12,D13,NC,32);
QEI encoder_m2(D10,D11,NC,32);

DigitalIn button(D2);

//Motor 3
BusOut motor_out(D0, D1, D2, D3);  // blue - pink - yellow - orange

//=========== OBJECTS USED ===================================================
// BiQuads
BiQuadChain bqc1;
BiQuadChain bqc2;
BiQuad bq1(0.98818943122,-1.59905034133, 0.98818943122, -1.59325743247, 0.98217188389); // notch
BiQuad bq2(0.80010174727,-1.60020349454, 0.80010174727, -1.55983681590, 0.64057017317); // high pass
//Take absolute value between high pass and low pass
BiQuad bq3(0.00035313141, 0.00070626282, 0.00035313141, -1.94614721828, 0.94755974392); // low pass

//States
enum state_t {REST0, REST, CALIBRATION, COUNTING, CONFIRMATION, GO2TABLESPOT, READY2CLEAN, HANDDOWN, CLEANING, CLEANCONFIRM, HANDUP, RETURN2REST};
state_t state = REST0;

//Tickers
Ticker stateTick;
Ticker screenTick;
Ticker angPos1;
Ticker t0;
Ticker t1;
Ticker t2;
Ticker t3;
Ticker t4;
Ticker t5;
Ticker t6;
Ticker t7;

Timer stateTimer;

//Go Flags
volatile bool fn0_go = false;
void fn0_activate(){ fn0_go = true; }; //Activates the go−flag
volatile bool fn1_go = false;
void fn1_activate(){ fn1_go = true; }; //Activates the go−flag
volatile bool fn2_go = false;
void fn2_activate(){ fn2_go = true; }; //Activates the go-flag
volatile bool fn3_go = false;
void fn3_activate(){ fn3_go = true; }; //Activates the go-flag
volatile bool fn4_go = false;
void fn4_activate(){ fn4_go = true; }; //Activates the go−flag
volatile bool fn5_go = false;
void fn5_activate(){ fn5_go = true; }; //Activates the go-flag
volatile bool fn6_go = false;
void fn6_activate(){ fn6_go = true; }; //Activates the go-flag
volatile bool fn7_go = false;
void fn7_activate(){ fn7_go = true; }; //Activates the go−flag  
volatile bool state_go = false;
void state_activate(){ state_go = true; }; 
volatile bool screen_go = false;
void screen_activate(){ screen_go = true; };

//=========== CONSTANTS =====================================================
const float ts = 0.005; //Sample time 200 Hz
const double pi = 3.14159265359;
const double transmissionShoulder = 94.4/40.2;
const double transmissionElbow = 1.0;

//Controller constants 
const double m1_Kp = 120.0, m1_Ki = 1.44876354368902, m1_Kd = -1.55261758822823, m1_N = 1.70578345077793;
const double m2_Kp = 120.0, m2_Ki = 1.44876354368902, m2_Kd = -1.55261758822823, m2_N = 1.70578345077793;
const double m1_Ts = 0.001; // Controller sample time motor 1 
const double m2_Ts = 0.001; // Controller sample time motor 2

//Angles
double O1 = 1.7463;
double O2 = 2.2183;
double O3 = 2.1194;
double O4 = 1.5004;
double O5 = 2.2240;
double O6 = 0.5323;
double B1 = 1.5381;
double B2 = 0.7378;
double B3 = 0.0082;
double B4 = -0.1243;
double B5 = 2.0296;
double B6 = 1.7815;

//Motor 3
int up = 1;
int down = 0;

//=========== BOOLEANS ======================================================
bool contracted = false;
bool confirm = false;
bool calibration = false;
bool EMG = true;

//=========== VARIABLES =====================================================
//Filtered EMG Signal
double emgFilteredRight;
double emgFilteredLeft;

//Max values
double emgMaxRight;
double emgMaxLeft;

//Setting the thresholds
double emgThresholdRightLow;
double emgThresholdRightHigh;
double emgThresholdLeftLow;
double emgThresholdLeftHigh;

int emgCounts;
int tableSpot;

//Timing aspect
int currentRepetitions = 0;
int previousRepetitions = 0;

//Motor 3
int motor3_direction = up; // direction
int begin_step = 0; 
int teller = 0;
double m3_pos_ts = 0;
double m3_clean_ts = 0;

//Motor variables
double m1_v1 = 0;
double m1_v2 = 0;
double m2_v1 = 0;
double m2_v2 = 0;

//Position variable
volatile double radians_m1;
volatile double radians_m2;

//Plant variable
volatile double motor1;
volatile double motor2;

//Reference positions
double reference_m1 = 0;
double reference_m2 = 0;

//=========== FUNCTIONS ======================================================

//=========== EMG ============================================================
void rest()
/*--------------------------------------------------
 This function is for the resting situation.
 --------------------------------------------------*/
{
    currentRepetitions = previousRepetitions+1;
    previousRepetitions = currentRepetitions;
}

void emgCount()
/*--------------------------------------------------
 This function counts the contractions of the right
 biceps.
 --------------------------------------------------*/
{
    if(emgFilteredRight >= emgThresholdRightHigh)
    {
        contracted = true;
    }
    
    if(contracted == true && emgFilteredRight <= emgThresholdRightLow)
    {
        contracted = false;
        emgCounts++;
    }
}

void emgConfirm()
/*--------------------------------------------------
 This function confirms the EMG counts via contracting
 the left biceps.
 --------------------------------------------------*/
{
    confirm = false;
    
    if(emgFilteredLeft >= emgThresholdLeftHigh)
    {
        contracted = true;
    }
    
    if(contracted == true && emgFilteredLeft <= emgThresholdLeftLow)
    {
        contracted = false;
        confirm = true;
    }
}

void emgFilterRight()
/*--------------------------------------------------
 This function filters and samples the EMG signal 
 from the right biceps.
 --------------------------------------------------*/
{
    double emgValueRight = emg1.read();
    double emgFilter1Right = bqc1.step(emgValueRight);
    double emgAbsRight = fabs(emgFilter1Right);
    
    emgFilteredRight = bqc2.step(emgAbsRight);
    
    if(state != CALIBRATION)
    {
        emgCount();  
    }
}

void emgFilterLeft()
/*--------------------------------------------------
 This function filters and samples the EMG signal 
 from the left biceps.
 --------------------------------------------------*/
{
    double emgValueLeft = emg2.read();
    double emgFilter1Left = bqc1.step(emgValueLeft);
    double emgAbsLeft = fabs(emgFilter1Left);
    
    emgFilteredLeft = bqc2.step(emgAbsLeft);
    
    if(state == CONFIRMATION || READY2CLEAN || CLEANCONFIRM)
    {
        emgConfirm();
    }
}

void emgMaxValueRight()
/*--------------------------------------------------
 This function measures the maximal EMG value to set
 thresholds for the right biceps. 
 It is used for calibration.
 --------------------------------------------------*/
{
    emgFilterRight();
    if(emgFilteredRight >= emgMaxRight)
    {
        emgMaxRight = emgFilteredRight;
    }
    
    //Setting the thresholds
    emgThresholdRightLow = 0.4*emgMaxRight;
    emgThresholdRightHigh = 0.7*emgMaxRight;
}

void emgMaxValueLeft()
/*--------------------------------------------------
 This function measures the maximal EMG value to set
 thresholds for the left biceps. 
 It is used for calibration.
 --------------------------------------------------*/
{
    emgFilterLeft();
    if(emgFilteredLeft >= emgMaxLeft)
    {
        emgMaxLeft = emgFilteredLeft;
    }
    
    //Setting the thresholds
    emgThresholdLeftLow = 0.4*emgMaxLeft;
    emgThresholdLeftHigh = 0.7*emgMaxLeft;
}

void emgCalibration()
/*--------------------------------------------------
 This function calibrates the EMG signal from the 
 right biceps.
 --------------------------------------------------*/
{
    currentRepetitions = previousRepetitions+1;
    previousRepetitions = currentRepetitions;
    
    if(currentRepetitions <= 1000)
    {
        emgMaxValueRight();
    }
    else if(1000 <= currentRepetitions && currentRepetitions <= 2000)
    {
        emgMaxValueLeft();
    }
    else if(currentRepetitions >= 2000)
    {
        calibration = true;
    }
}

void getTableSpot()
/*--------------------------------------------------
 This function determines to which spot on the table
 the robot arm should move.
 --------------------------------------------------*/
{
    currentRepetitions = previousRepetitions+1;
    previousRepetitions = currentRepetitions;
     
    if(200 <= currentRepetitions && currentRepetitions < 2000)
    {
        emgFilterRight();
        tableSpot = emgCounts;
    }
}

//=========== MOTOR ==========================================================
double PID1( double err, const double Kp, const double Ki, const double Kd,
                const double Ts, const double N, double &v1, double &v2 ) 
/*--------------------------------------------------
 This function is a PID controller for motor 1.
 --------------------------------------------------*/
{
    // These variables are only calculated once!
    const double a1 = (-4.0/(N*Ts+2));
    const double a2 = -(N*Ts-2)/(N*Ts+2);
    const double b0 = (4.0*Kp + 4*Kd*N + 2*Ki*Ts + 2*Kp*N*Ts + Ki*N*pow(Ts,2))/(2*N*Ts + 4);
    const double b1 = (Ki*N*pow(Ts,2) - 4.0*Kp - 4.0*Kd*N)/(N*Ts + 2.0);
    const double b2 = (4*Kp + 4*Kd*N - 2*Ki*Ts - 2*Kp*N*Ts + Ki*N*pow(Ts,2))/(2*N*Ts + 4);
 
    double v = err - a1*v1 - a2*v2;
    double u = b0*v + b1*v1 + b2*v2; 
    v2 = v1; v1 = v;
    return u;
}
 
double PID2( double err, const double Kp, const double Ki, const double Kd,
                const double Ts, const double N, double &v1_2, double &v2_2 ) 
/*--------------------------------------------------
 This function is a PID controller for motor 1.
 --------------------------------------------------*/
{
    // These variables are only calculated once!
    const double a1 = (-4.0/(N*Ts+2));
    const double a2 = -(N*Ts-2)/(N*Ts+2);
    const double b0 = (4.0*Kp + 4*Kd*N + 2*Ki*Ts + 2*Kp*N*Ts + Ki*N*pow(Ts,2))/(2*N*Ts + 4);
    const double b1 = (Ki*N*pow(Ts,2) - 4.0*Kp - 4.0*Kd*N)/(N*Ts + 2.0);
    const double b2 = (4*Kp + 4*Kd*N - 2*Ki*Ts - 2*Kp*N*Ts + Ki*N*pow(Ts,2))/(2*N*Ts + 4);
 
    double v = err - a1*v1_2 - a2*v2_2;
    double u = b0*v + b1*v1_2 + b2*v2_2; 
    v2_2 = v1_2; v1_2 = v;
    return u;
}
 
void getAngPosition_m1() 
/*--------------------------------------------------
 This function gets the angular position of motor 1.
 --------------------------------------------------*/
{   
    volatile int pulses_m1 = encoder_m1.getPulses();
    radians_m1 = (pulses_m1*0.002991*0.5); //2 = encoding type, 3591.84 = counts per revoluton for the two big motors
}
 
void getAngPosition_m2()
/*--------------------------------------------------
 This function gets the angular position of motor 2.
 --------------------------------------------------*/
{   
    volatile int pulses_m2 = encoder_m2.getPulses();
    radians_m2 = (pulses_m2*0.002991*0.5); //2 = encoding type, 3591.84 = counts per revoluton for the two big motors  
}
  
void motor1_Controller(double radians_m1)
/*--------------------------------------------------
 This function measures the error and applies it to
 the output to the plant.
 --------------------------------------------------*/
{
    volatile double error_m1 = reference_m1 - radians_m1;
    motor1 = PID1( error_m1,m1_Kp,m1_Ki,m1_Kd,m1_Ts, m1_N, m1_v1, m1_v2 );
}
 
  // Next task, measure the error and apply the output to the plant
void motor2_Controller(double radians_m2)
/*--------------------------------------------------
 This function measures the error and applies it to
 the output to the plant.
 --------------------------------------------------*/
{
    volatile double error_m2 = reference_m2 - radians_m2;
    motor2 = PID1( error_m2,m2_Kp,m2_Ki,m2_Kd,m2_Ts, m2_N, m2_v1, m2_v2 );
 
  //  pc.printf("motor2 = %d",reference_m2);
}
    
void control_m1(double motor1, double radians_m1)
/*--------------------------------------------------
 This function controls motor 1.
 --------------------------------------------------*/
{
    if(abs(motor1) > 1.0)
    {
        motor1MagnitudePin=0.6;//MOET NOG TUSSENWAAREN KRIJGEN
    }
    else
    {
        motor1MagnitudePin=0.0;    
    }
    
    if(motor1 <= 0)      
    {
        motor1DirectionPin=0.0;
    }
    else    
    {
        motor1DirectionPin=1.0;
    } 
}
 
void control_m2(double motor2, double radians_m2)
/*--------------------------------------------------
 This function controls motor 2.
 --------------------------------------------------*/
{
    if(abs(motor2) > 1)
    {
        motor2MagnitudePin=0.5;///MOET NOG TUSSENWAAREN KRIJGEN
    }
    else
    {
        motor2MagnitudePin=0.0;    
    }
    
    if(motor2 <= 0)    
    {
        motor2DirectionPin=1.0;
    }
    else    
    {
        motor2DirectionPin=0.0;
    }
}

void motor3_position()
{
/*--------------------------------------------------
 This function controls motor 3 to go up and down.
 --------------------------------------------------*/
    switch(begin_step)
        { 
            case 0: motor_out = 0x1; break;  // 0001
            case 1: motor_out = 0x3; break;  // 0011
            case 2: motor_out = 0x2; break;  // 0010   
            case 3: motor_out = 0x6; break;  // 0110
            case 4: motor_out = 0x4; break;  // 0100
            case 5: motor_out = 0xC; break;  // 1100
            case 6: motor_out = 0x8; break;  // 1000
            case 7: motor_out = 0x9; break;  // 1001
            
            default: motor_out = 0x0; break; // 0000
        }
  
        if(motor3_direction == up)
        { 
            begin_step++;
            teller++;
        } 
        else
        { 
            begin_step--; 
            teller--;
        }
        
        if(begin_step>7)begin_step=0; 
        
        if(begin_step<0)begin_step=7; 
        
        /*
        if(teller <= -100000)
        {
            m3_pos_ts = 0;
        }
        
        if(teller => 0)
        {
            m3_pos_ts = 0;
        }
        */
}

void motor3_clean()
/*--------------------------------------------------
 This function controls motor 3, makes it go back \
 and forth.
 --------------------------------------------------*/
{
    switch(begin_step)
        { 
            case 0: motor_out = 0x1; break;  // 0001
            case 1: motor_out = 0x3; break;  // 0011
            case 2: motor_out = 0x2; break;  // 0010   
            case 3: motor_out = 0x6; break;  // 0110
            case 4: motor_out = 0x4; break;  // 0100
            case 5: motor_out = 0xC; break;  // 1100
            case 6: motor_out = 0x8; break;  // 1000
            case 7: motor_out = 0x9; break;  // 1001
            
            default: motor_out = 0x0; break; // 0000
        }
  
        if(motor3_direction==1)
        { 
            begin_step++;
            teller++;
        } 
         
        else
        { 
            begin_step--; 
            teller--;
        }
        
        if(begin_step>7)begin_step=0; 
        
        if(begin_step<0)begin_step=7;
        
        if(teller <= -11000)
        {
            motor3_direction = up;
        }
        
        if(teller >= -9000)
        {
            motor3_direction = down;
        }
}

//=========== STATE ==========================================================
void stateMachine()
/*--------------------------------------------------
 This function makes sure different functions are
 done in a certain order.
 --------------------------------------------------*/
{
    stateTimer.reset();
    switch(state)
    {
        case REST0:
        {
            pc.printf("Hi I am Claire!\r\n");
            state = REST;
        }
        break;
        
        case REST:
        {
            rest();
            if(currentRepetitions >= 1000 && EMG == 1)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = CALIBRATION;
            }
            else if(currentRepetitions >= 1000)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                emgCounts = 0;
                state = COUNTING;
            }
        }
        break;
        
        case CALIBRATION: 
        {
            emgCalibration();
            if(currentRepetitions >=2200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = COUNTING;
                EMG = 0;
            }
        }
        break;
        
        case COUNTING:
        {
            getTableSpot();
            if(currentRepetitions >= 2200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = CONFIRMATION;
            }
        }
        break;
        
        case CONFIRMATION:
        {
            rest(); //For counting purposes
            emgFilterLeft();
            if(confirm == true && currentRepetitions >= 1200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = GO2TABLESPOT;
            }
            else if(currentRepetitions >= 1200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = REST;
            }
        }
        break;
        
        case GO2TABLESPOT:
        {
            rest(); //For counting purposes
            if(currentRepetitions < 2200) //Tijd = 10 secondes, misschien anders.
                if(tableSpot == 1)
                {
                    reference_m1 = O1*transmissionElbow;
                    reference_m2 = B1*transmissionShoulder;
                }
                
                else if(tableSpot == 2)
                {
                    reference_m1 = O2*transmissionElbow;
                    reference_m2 = B2*transmissionShoulder;
                }
                
                else if(tableSpot == 3)
                {
                    reference_m1 = O3*transmissionElbow;
                    reference_m2 = B3*transmissionShoulder;
                }
                
                else if(tableSpot == 4)
                {
                    reference_m1 = O4*transmissionElbow;
                    reference_m2 = B4*transmissionShoulder;
                }
                
                else if(tableSpot == 5)
                {
                    reference_m1 = O5*transmissionElbow;
                    reference_m2 = B5*transmissionShoulder;
                }
                
                else if(tableSpot == 6)
                {
                    reference_m1 = O6*transmissionElbow;
                    reference_m2 = B6*transmissionShoulder;
                }
               
                else
                {
                    if(currentRepetitions == 200)
                    {
                        currentRepetitions = 0;
                        previousRepetitions = 0;
                        tableSpot = 0;
                        state = COUNTING;
                    }
                }
                    
            if(currentRepetitions >= 2200 && tableSpot >= 1 && tableSpot <= 6 )
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = CLEANING;
            }
        }
        break;
        
        case READY2CLEAN:
        {
            rest(); //For counting purposes
            emgFilterLeft();
            
            if(confirm == true && currentRepetitions >= 1200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = HANDDOWN;
            }
            else if(currentRepetitions >= 1200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = RETURN2REST;
            }
        }
        break;
        
        case HANDDOWN:
        {
            m3_pos_ts = 0.0014;
            motor3_direction = down;
            
            if(teller == -10000)
            {
               m3_pos_ts = 100000;
               state = CLEANING;
            }
        }
        break;
               
        case CLEANING:
        {
            rest(); //For counting purposes
            m3_clean_ts = 0.0014;
            if(currentRepetitions >= 100)
            {
                reference_m1+=0.1;
                reference_m2+=0.1;
            }
            else if(currentRepetitions >= 500)
            {
                reference_m1-=0.2;
                reference_m2-=0.2;
            }
            else if(currentRepetitions >= 900)
            {
                reference_m1+=0.2;
                reference_m2+=0.2;
            }
            else if(currentRepetitions >= 1300)
            {
                reference_m1-=0.2;
                reference_m2-=0.2;
            }
            else if(currentRepetitions >= 1700)
            {
                reference_m1+=0.2;
                reference_m2+=0.2;
            }
            
            if(currentRepetitions >= 2050)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                m3_clean_ts = 0;
                state = CLEANCONFIRM;
            }
        }
        break;
        
        case CLEANCONFIRM:
        {
            rest();
            emgFilterLeft();
            
            if(confirm == true && currentRepetitions >= 1200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = HANDUP;
            }
            else if(currentRepetitions >= 1200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                state = CLEANING;
            }
        }
        break;
        
        case HANDUP:
        {
            m3_pos_ts = 0.0014;
            motor3_direction = up;
            
            if(teller == 0)
            {
               m3_pos_ts = 100000;
               state = RETURN2REST;
            }
        }
        break;
            
        case RETURN2REST:
        {
            rest(); //For counting purposes
            if(currentRepetitions < 2200)
            {
                reference_m1 = 0;
                reference_m2 = 0;
            }
            
            if(currentRepetitions >= 2200)
            {
                currentRepetitions = 0;
                previousRepetitions = 0;
                tableSpot = 0;
                state = REST;
            }                
        }
        break;
        
        default: 
        {
            currentRepetitions = 0;
            previousRepetitions = 0;
            state = REST;
        } 
    }
}

void screenUpdate()
/*--------------------------------------------------
 This function prints the commands to the screen.
 --------------------------------------------------*/
{ 
    if(state == CALIBRATION)
    {
        if(currentRepetitions == 100)
        {
             pc.printf("Welcome to calibration.\n\rYou have 5 seconds to contract your right biceps.\n\r");
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("1\n\r");
        }
        else if(currentRepetitions == 1050)
        {
            pc.printf("You have 5 seconds to contract your left biceps.\n\r");
        }
        else if(currentRepetitions == 1200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 1400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 1600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 1800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 2000)
        {
            pc.printf("1\n\r");
        }
        else if(currentRepetitions == 2050)
        {
            pc.printf("Calibrations is complete.\n\r");
            /*
            pc.printf("ThresholdRightHigh = %f\n\r.",emgThresholdRightHigh);
            pc.printf("ThresholdLeftHigh = %f\n\r.",emgThresholdLeftHigh);
            pc.printf("ThresholdRightLow = %f\n\r.",emgThresholdRightLow);
            pc.printf("ThresholdLeftLow = %f\n\r.",emgThresholdLeftLow);
            */
        }
    }

    else if(state == COUNTING)
    {
        if(currentRepetitions == 100)
        {
            pc.printf("You have 10 seconds to contract your right biceps x times to go to that spot on the table.\n\r");
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("10\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("9\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("8\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("7\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("6\n\r");
        }
        else if(currentRepetitions == 1200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 1400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 1600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 1800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 2000)
        {
            pc.printf("1\n\r");
        }
        else if(currentRepetitions == 2100)
        {
            pc.printf("You have selected table spot nr %i, is this correct?\n\r",tableSpot);
        }
    } 

    else if(state == CONFIRMATION)
    {
        if(currentRepetitions == 100)
        {
             pc.printf("You have 5 seconds to confirm with your left biceps.\n\r");
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("1\n\r");
        }
        else if(confirm == true && currentRepetitions == 1050)
        {
            pc.printf("Claire will go to table spot nr. %i now.\n\r", tableSpot);
        }
        else if(confirm == false && currentRepetitions == 1050)
        {
            pc.printf("You have cancelled.\n\r");
        }
    }
    
    else if(state == GO2TABLESPOT)
    {
        if(tableSpot >= 1 && tableSpot <= 6)
        {
        if(currentRepetitions == 100)
        {  
            pc.printf("Claire is going to table spot nr. %i.\n\r",tableSpot);
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("10\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("9\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("8\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("7\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("6\n\r");
        }
        else if(currentRepetitions == 1200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 1400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 1600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 1800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 2000)
        {
            pc.printf("1\n\r");
        }
        else if(currentRepetitions == 2050)
        {
            pc.printf("Claire is at table spot nr. %i. It will be cleaned shortly.\n\r",tableSpot);
        }
        }
        
        else
        {
            if(currentRepetitions == 100)
            {
                pc.printf("This isn't a valid table spot. Please try again.");  
            }
        }     
    }
    
    else if(state == READY2CLEAN)
    {
        if(currentRepetitions == 100)
        {
             pc.printf("You have 5 seconds to confirm with your left biceps that you want to clean the table spot Claire is on.\n\r");
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("1\n\r");
        }
        else if(confirm == true && currentRepetitions == 1050)
        {
            pc.printf("Table spot nr. %i will be cleaned now.\n\r", tableSpot);
        }
        else if(confirm == false && currentRepetitions == 1050)
        {
            pc.printf("You have cancelled.\n\r");
        }
    }
    
    else if(state == HANDDOWN)
    {
        if(currentRepetitions == 100)
        {
            pc.printf("Claire's hand is going down to the table now. \n\r");
        }
    }
    
    
    else if(state == CLEANING)
    {
        if(currentRepetitions == 100)
        {
            pc.printf("Claire is cleaning table spot nr. %i. It will be done in\n\r.",tableSpot);
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("10\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("9\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("8\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("7\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("6\n\r");
        }
        else if(currentRepetitions == 1200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 1400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 1600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 1800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 2000)
        {
            pc.printf("1\n\r");
        }
        else if(currentRepetitions == 2050)
        {
            pc.printf("Claire has cleaned table spot nr. %i.\n\r",tableSpot);
        }
    }
    
    else if(state == CLEANCONFIRM)
    {
        if(currentRepetitions == 100)
        {
            pc.printf("Is table spot %i cleaned? Contract your left biceps to confirm.\n\r",tableSpot);
        }
        else if(currentRepetitions == 200)
        {
            pc.printf("5\n\r");
        }
        else if(currentRepetitions == 400)
        {
            pc.printf("4\n\r");
        }
        else if(currentRepetitions == 600)
        {
            pc.printf("3\n\r");
        }
        else if(currentRepetitions == 800)
        {
            pc.printf("2\n\r");
        }
        else if(currentRepetitions == 1000)
        {
            pc.printf("1\n\r");
        }
        else if(confirm == true && currentRepetitions == 1050)
        {
            pc.printf("Very good! Claire will return to the starting position.\n\r");
        }
        else if(confirm == false && currentRepetitions == 1050)
        {
            pc.printf("Claire will clean table spot nr. %i again.\n\r", tableSpot);
        }
    }
    
    else if(state == HANDUP)
    {
        if(currentRepetitions == 100)
        {
            pc.printf("Claire's hand is going up now. \n\r");
        }
    } 
    
    else if(state == RETURN2REST)
    {
        if(currentRepetitions == 100)
        {
            pc.printf("Claire is going back to the start. Which table spot do you want to clean next?\r\n");
        }
    }   
}

//=========== MAIN ===========================================================
int main()
{
    pc.baud(115200);
    
    bqc1.add(&bq1).add(&bq2);
    bqc2.add(&bq3);
    
    state = REST0;
    stateTick.attach(&state_activate,ts);
    screenTick.attach(&screen_activate,ts);
    
    motor1MagnitudePin.period(1.0/1000.0);
    motor2MagnitudePin.period(1.0/1000.0);
    t1.attach(&fn1_activate, 0.0001f);
    t2.attach(&fn2_activate, 0.0001f);
    t3.attach(&fn3_activate, 0.0001f);
    t4.attach(&fn4_activate, 0.0001f);
    t5.attach(&fn5_activate, 0.0001f);
    t6.attach(&fn6_activate, 0.0001f);
    
    //Motor 3
    t0.attach(&fn0_activate, m3_pos_ts);
    t7.attach(&fn7_activate, m3_clean_ts);

    while(1)
    {       
        if(fn1_go)
        {
            fn1_go = false;
            getAngPosition_m1();
        }
        
        if(fn2_go)
        {
            fn2_go = false;
            motor1_Controller(radians_m1);
        }
        
        if(fn3_go)
        {
            fn3_go = false;
            control_m1(motor1,radians_m1);
        }
        
        if(fn4_go)
        {
            fn4_go = false;
            getAngPosition_m2();
        }
        
        if(fn5_go)
        {
            fn5_go = false;
            motor2_Controller(radians_m2);
        }
        
        if(fn6_go)
        {
            fn6_go = false;
            control_m2(motor2,radians_m2);
        }
        
        if(fn0_go)
        {
            fn0_go = false;
            motor3_position();
        }
        
        if(fn7_go)
        {
            fn7_go = false;
            motor3_clean();
        }
        
        if(state_go)
        {
            state_go = false;
            stateMachine();
        }
        
        if(screen_go)
        {
            screen_go = false;
            screenUpdate();
        }
    }
}   
