/// SPDX-License-Indentifier: ISC
/// Copyright: 2020+ Philippe Coval <https://purl.org/rzr/>
/// URL: https://os.mbed.com/users/rzrfreefr/code/rzr-example-mbed/

#include "mbed.h"

#include <EthernetInterface.h>
#include <USBKeyboard.h>
#include <FXOS8700Q.h>
#include "Utils.h"

//#define CONFIG_LOG 1
//#define CONFIG_NET 1
#define CONFIG_USB 1

#if defined(CONFIG_LOG) && CONFIG_LOG
# define logf printf
#else
# define logf if (false) printf
#endif


int main()
{
    logf("#{ %s\n", __FILE__);

    DigitalOut led(LED1);
    led = false; //on

    I2C i2c(PTE25, PTE24);
    motion_data_units_t acc_data;
    motion_data_counts_t acc_raw;

    float vorigin[] = {0, 0, 0};
    float vacc[] = {0, 0, 0};
    const int len = 16;
    float vhist[len][3];

    float vmean[] = {0, 0, 0};
    float vmin[] = {0, 0, 0};
    float vmax[] = {0, 0, 0};

    // cis: 10 < 12 < 13Log < 15 < 20N < 25<  35N < 40 < ~50 < 60 < !75
    // pin: 12? < 50
    float threshold = 1./25;
    float vthreshold[] = {threshold, threshold, threshold};
    const float delay = 1000.f / 25 / len;

#if 0
    int keymap[3][2] = {
        { RIGHT_ARROW, LEFT_ARROW },
        { DOWN_ARROW, UP_ARROW },
        { KEY_PAGE_DOWN, KEY_PAGE_UP }
    };
#elif 1
    int keymap[3][2] = {
        { KEY_RCTRL, KEY_CTRL },
        { DOWN_ARROW, UP_ARROW },
        { KEY_PAGE_DOWN, KEY_PAGE_UP }
    };
#elif 0 // https://github.com/ARMmbed/mbed-os/blob/c6094f7b36dc0e90a6a7271870333fbba475286c/drivers/usb/source/USBKeyboard.cpp
    int keymap[3][2] = {
        { 0x4f, 0x50 },
        { 0x51, 0x52 },
        { 0x4b, 0x4e }
    };
#endif


    FXOS8700QAccelerometer acc(i2c, FXOS8700CQ_SLAVE_ADDR1); // Proper Ports and I2C Address for K64F Freedom board
#if defined(CONFIG_USB) && CONFIG_USB
    logf("# usb:\n");
    USBKeyboard usb; ///< https://os.mbed.com/docs/mbed-os/v6.2/apis/usbkeyboard.html
#endif

#if defined(CONFIG_NET) && CONFIG_NET
    EthernetInterface eth; ///< https://os.mbed.com/docs/mbed-os/v6.2/apis/ethernet.html
    eth.set_dhcp(true);
    eth.connect();
    logf("\n# MAC address: %s\n", eth.get_mac_address());
    SocketAddress a;
    eth.get_ip_address(&a);
    char* addr = a.get_ip_address() || "127.0.0.1";
    logf("# IP address: %s\n", addr);
#endif

    acc.enable();
    acc.getAxis(acc_data);
    vorigin[0] = acc_data.x;
    vorigin[1] = acc_data.y;
    vorigin[2] = acc_data.z;

    logf("\r\n\n# FXOS8700Q: Who Am I= %X\r\n", acc.whoAmI());

    int key = 0;
    for(int iter=0; iter<len; ++iter) {
        if (false) led = !led;
        acc.getAxis(acc_data);
        vacc[0] = acc_data.x - vorigin[0];
        vacc[1] = acc_data.y - vorigin[1];
        vacc[2] = acc_data.z - vorigin[2];
        logf("{\"accel\":[%.2f,%.2f,%.2f]}\r\n", vacc[0], vacc[1], vacc[2]);

#if defined(CONFIG_USB) && CONFIG_USB
        usb.connect();

        if (key == 0) {
            for(int i=2; i>=0; i--) {
                if (vacc[i] > vthreshold[i]) {
                    key = keymap[i][0];
                } else if (vacc[i] < -vthreshold[i]) {
                    key = keymap[i][1];
                }
            }

            if (key != 0) {
                led = false; //on
                logf("# key: %0X\n", key);
#if 0
                Thread thread;
                thread.start(idle);
#endif

                if (!true) {
                    usb.key_code(key, key);
                } else if (false) {
                    usb.key_code(key);
                } else {
                    Utils::keyboard_key_code(usb, key, key);
                }
            }
            iter=0;
            continue;
        }
#endif

        for (int i=0; i<3; i++) {
            vhist[iter][i] = vacc[i];
        }

        if (iter==len-1) {
            led = false; // heart beat
            led = true;

            for (int i=0; i<3; i++) {
                vmean[i] = 0;
                vmin[i] = vmax[i] = vhist[iter][i];
            }

            for (iter=0; iter<len; iter++) {
                for (int i=0; i<3; i++) {
                    vmean[i] += vhist[iter][i];
                    if (vmin[i] > vhist[iter][i]) {
                        vmin[i] = vhist[iter][i];
                    }
                    if (vmax[i] < vhist[iter][i]) {
                        vmax[i] = vhist[iter][i];
                    }
                }
            }
            for (int i=0; i<3; i++) {
                vmean[i] /= len;
            }

            bool reset = true;
            for (int k=0; k<3; k++) {
                if (abs(vmax[k] - vmin[k]) > threshold*.8) {
                    reset &= false;
                }
            }
            if (reset) {
                logf("\r\n# Reset origin\n");
                vorigin[0] = acc_data.x;
                vorigin[1] = acc_data.y;
                vorigin[2] = acc_data.z;

                key = 0; // unbounce
                led = true; //off
            }
            iter=0;
        }
        ThisThread::sleep_for(delay);
    }
    logf("#} %s\n", __FILE__);
}