#include "mbed.h"
#include "rtos.h"

#define UART_TRACE  (0)
#include "SpiSequenceSender.h"
#include "EnvelopeGenerator.h"
#include "SpiAmpController.h"
#include "SpiFilterController.h"

#define SEQUENCE_N  (16)
#define SPI_RATE    (8000000)

const int samplingPeriod = 1;   // ms
const int bpm = 120;
const int envelopeLength = (60 * 1000 / (bpm * 4)) / samplingPeriod;
const int waveShape = SpiSequenceSender::WAVESHAPE_SAW;
const int baseNoteNumber = 48;

// Initial Sequence
const int noteOn[SEQUENCE_N] = { 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 0 };
const int octave[SEQUENCE_N] = {-1,-1,-1, 0, 0,-1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0 };
const int pitch[SEQUENCE_N]  = { 9, 7, 3, 0, 0, 3, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0 };
const int tie[SEQUENCE_N]    = { 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0 };

//AnalogIn levelIn(A0);
AnalogIn durationIn(A2);
AnalogIn decayIn(A1);
AnalogIn sustainIn(A0);
AnalogIn cutoffIn(A3);
AnalogIn resonanceIn(A4);

SPI spiMaster(SPI_MOSI, SPI_MISO, SPI_SCK);

Sequence sequence[SEQUENCE_N];
SpiSequenceSender sequenceSender(&spiMaster, D9, sequence, SEQUENCE_N, samplingPeriod, bpm);

Envelope envelope(4095, envelopeLength, envelopeLength*3/4, envelopeLength/2, 2047);
EnvelopeGenerator envelopeGenerator;
SpiAmpController ampController(&spiMaster, D8, D7);

SpiFilterController filterController(&spiMaster, D10);

void callbackFunction(int ticks)
{
    if (ticks == 0) {
        envelopeGenerator.init(envelope);
    }

    if (sequenceSender.getSequences()[sequenceSender.getStep()].noteOn) {
        uint16_t level = envelopeGenerator.getModLevel();
        ampController.outDca(level);
    } else {
        ampController.outDca(0);
    }
    envelopeGenerator.update();
    
    filterController.outDCF();
}

int main()
{
    spiMaster.format(8, 0);
    spiMaster.frequency(SPI_RATE);
    
    // Test SequencerSender Run
    //
    Sequence::baseNoteNumber = baseNoteNumber;
    sequenceSender.setBpm(bpm);
    
    for (int i = 0; i < SEQUENCE_N; i++) {
        Sequence& seq = sequenceSender.getSequences()[i];
        seq.setPitch(pitch[i]);
        seq.setOctave(octave[i]);
        seq.noteOn = noteOn[i];
        seq.tie = tie[i];
    }
    
    envelopeGenerator.init(envelope);

    sequenceSender.attachUpdate(&callbackFunction);
    sequenceSender.setWaveShape(waveShape);
    sequenceSender.run(0);
    
    for (;;) {
        /*
        sequenceSender.setPulseWidth(sequenceSender.getPulseWidth() + 4);
        Thread::wait(500);
        sequenceSender.setWaveShape(SpiSequenceSender::WAVESHAPE_SAW);
        Thread::wait(500);
        sequenceSender.setWaveShape(SpiSequenceSender::WAVESHAPE_SQUARE);
        */
        //envelope.setLevel(levelIn * 4095);
        envelope.setLevel(4095);
        envelope.setDuration(durationIn * envelopeLength);
        envelope.setDecay(decayIn * envelopeLength);
        envelope.setSustain(sustainIn * 4095);
        
        filterController.setCutoff(cutoffIn * 255);
        filterController.setResonance(resonanceIn * 255);
        
        #if(UART_TRACE)
        printf("%d\t%d\t%d\t%d\t%d\t%d\r\n",
            filterController.getCutoff(),
            filterController.getResonance(),
            envelope.getLevel(),
            envelope.getDuration(),
            envelope.getDecay(),
            envelope.getSustain()
        );
        #endif
    }
}
