/*
 * KIK01
 * Kick Machine
 *
 * 2017.06.04 created.
 *
 */

#include "mbed.h"
#include "rtos.h"

#define UART_TRACE      (1)
#define TITLE_STR1      ("KIK01 Kick Machine")
#define TITLE_STR2      ("20170606")

#define PI_F            (3.1415926f)
#define SAMPLING_RATE   (96000)
#define SAMPLING_PERIOD (1.0f/SAMPLING_RATE)
#define UPDATE_RATE     (32000)

AnalogOut Dac1(PA_5);

AnalogIn Ain0(PA_0);
AnalogIn Ain1(PA_1);
AnalogIn Ain2(PA_4);
AnalogIn Ain3(PB_0);
AnalogIn Ain4(PC_1);
AnalogIn Ain5(PC_0);
AnalogIn Ain6(PC_2);
AnalogIn Ain7(PC_3);

AnalogIn Ain8(PC_4);
AnalogIn Ain9(PB_1);
AnalogIn Ain10(PC_5);
AnalogIn Ain11(PA_6);

class EnvelopeAR
{
public:
    EnvelopeAR(int _attack, int _release, float _v0, float _v1, float _v2, float _attackTauRatio=0.36f, float _releaseTauRatio=0.36f) :
        amplitude(_v0),
        v0(_v0),
        v1(_v1),
        v2(_v2),
        vLast(_v0),
        attackTauRatio(_attackTauRatio),
        releaseTauRatio(_releaseTauRatio) {
        setAttack(_attack);
        setRelease(_release);
    }

    ~EnvelopeAR() {}

    void setAttack(int _attack) {
        attack = _attack;
        tau0 = attack * attackTauRatio;
    }
    int getAttack() {
        return attack;
    }

    void setRelease(int _release) {
        release = _release;
        tau1 = release * releaseTauRatio;
    }
    int getRelease() {
        return release;
    }

    void setAttackTauRatio(float _attackTauRatio) {
        attackTauRatio = _attackTauRatio;
        tau0 = attack * attackTauRatio;
    }
    float getAttackTauRatio() {
        return attackTauRatio;
    }

    void setReleaseTauRatio(float _releaseTauRatio) {
        releaseTauRatio = _releaseTauRatio;
        tau1 = release * releaseTauRatio;
    }
    float getReleaseTauRatio() {
        return releaseTauRatio;
    }

    float getTau0() {
        return tau0;
    }
    float getTau1() {
        return tau1;
    }

    void setV0(float _v0) {
        v0 = _v0;
    }
    float getV0() {
        return v0;
    }
    void setV1(float _v1) {
        v1 = _v1;
    }
    float getV1() {
        return v1;
    }
    void setV2(float _v2) {
        v2 = _v2;
    }
    float getV2() {
        return v2;
    }

    float getAmplitude() {
        return amplitude;
    }
    float getAmplitude(int tick) {
        if (tick < attack) {
            // attackの処理
            amplitude = v0 + (v1 - v0) * (1 - expf(-(float)tick / tau0));
            vLast = amplitude;
        } else {
            // releaseの処理
            amplitude = (vLast - v2) * (expf(-(float)(tick - attack) / tau1)) + v2;
        }
        return amplitude;
    }

private:
    int attack;
    int release;
    float amplitude;
    float v0;
    float v1;
    float v2;
    float vLast;
    float tau0;
    float tau1;
    float attackTauRatio;
    float releaseTauRatio;
};

class EnvelopeParam
{
public:
    int attack;
    int release;
    float v0;
    float v1;
    float v2;
    float attackTauRatio;
    float releaseTauRatio;
};

EnvelopeAR envelopeFrequency(5, 300, 880.0f, 120.0f, 40.0f, 0.36f, 0.1f);
EnvelopeAR envelopeAmplitude(50, 200, 0.99f, 1.0f, 0.0f);

volatile EnvelopeParam frequencyParam;
volatile EnvelopeParam amplitudeParam;

volatile int ticks;
volatile int envelopeTicks;
volatile float frequency;
volatile float phi;
volatile float phiDelta;
volatile float amplitude;

volatile float bpm;
volatile int envelopeLength;

void generateWave()
{
    phi += phiDelta;
    if (phi >= 1.0f) {
        phi -= 2.0f;
    }
    float level = cosf(PI_F * phi) * amplitude;

    Dac1.write((level * 0.7f + 1.0f) / 2.0f);
}

void generateEnvelope()
{
    // Frequency Envelope
    frequency = envelopeFrequency.getAmplitude(envelopeTicks);
    phiDelta = 2.0f * frequency / SAMPLING_RATE;

    // Amplitude Envelope
    amplitude = envelopeAmplitude.getAmplitude(envelopeTicks);

    envelopeTicks++;
    if (envelopeTicks >= envelopeLength) {
        envelopeTicks = 0;
    }
}

void update()
{
    ticks++;
    if (ticks >= SAMPLING_RATE / UPDATE_RATE) {
        ticks = 0;

        // set envelope parameters
        envelopeAmplitude.setAttack(amplitudeParam.attack);
        envelopeAmplitude.setRelease(amplitudeParam.release);
        envelopeAmplitude.setV0(amplitudeParam.v0);
        envelopeAmplitude.setV1(amplitudeParam.v1);
        envelopeAmplitude.setV2(amplitudeParam.v2);
        envelopeAmplitude.setAttackTauRatio(amplitudeParam.attackTauRatio);
        envelopeAmplitude.setReleaseTauRatio(amplitudeParam.releaseTauRatio);

        envelopeFrequency.setAttack(frequencyParam.attack);
        envelopeFrequency.setRelease(frequencyParam.release);
        envelopeFrequency.setV0(frequencyParam.v0);
        envelopeFrequency.setV1(frequencyParam.v1);
        envelopeFrequency.setV2(frequencyParam.v2);
        envelopeFrequency.setAttackTauRatio(frequencyParam.attackTauRatio);
        envelopeFrequency.setReleaseTauRatio(frequencyParam.releaseTauRatio);

        generateEnvelope();
    }
    generateWave();
}

void setParams()
{
    bpm = Ain0.read() * 180.0f + 60.0f;
    envelopeLength = 60 * UPDATE_RATE / bpm;

    amplitudeParam.attack = Ain1.read() * envelopeLength;
    amplitudeParam.release = Ain2.read() * envelopeLength;
    amplitudeParam.v0 = Ain3.read();
    amplitudeParam.v1 = 1.0f;
    amplitudeParam.v2 = 0.0f;
    amplitudeParam.attackTauRatio = 0.36f;
    amplitudeParam.releaseTauRatio = Ain4.read() + 0.01f;

    frequencyParam.attack = Ain8.read() * envelopeLength * 0.1f;
    frequencyParam.release = Ain11.read() * envelopeLength + 1;
    frequencyParam.v0 = Ain5.read() * 4000.0f;
    frequencyParam.v1 = Ain6.read() * 400.0f;
    frequencyParam.v2 = Ain7.read() * 400.0f;
    frequencyParam.attackTauRatio = Ain9.read() + 0.01f;
    frequencyParam.releaseTauRatio = Ain10.read() + 0.01f;
}

int main()
{
    printf("%s %s\r\n", TITLE_STR1, TITLE_STR2);

    frequency = 1000.0f;
    phiDelta = 2.0f * frequency / SAMPLING_RATE;
    amplitude = 1.0f;

    ticks = 0;
    envelopeTicks = 0;

    bpm = 120.0f;
    setParams();

    Ticker samplingTicker;
    samplingTicker.attach(&update, SAMPLING_PERIOD);

    for (;;) {
        setParams();

#if UART_TRACE
        printf("%.1f\t%d\t", bpm, envelopeLength);

        printf("%d\t%d\t", amplitudeParam.attack, amplitudeParam.release);
        printf("%.2f\t%.2f\t%.2f\t", amplitudeParam.v0, amplitudeParam.v1, amplitudeParam.v2);
        printf("%.2f\t%.2f\t", amplitudeParam.attackTauRatio, amplitudeParam.releaseTauRatio);

        printf("%d\t%d\t", frequencyParam.attack, frequencyParam.release);
        printf("%.2f\t%.2f\t%.2f\t", frequencyParam.v0, frequencyParam.v1, frequencyParam.v2);
        printf("%.2f\t%.2f\r\n", frequencyParam.attackTauRatio, frequencyParam.releaseTauRatio);
#endif

        Thread::wait(100);
    }
}
