/*
 * main.cpp
 * BaseMachine Sequencer
 *
 * 2016.08.17
 *
 */
 
#define UART_TRACE  (1)

#include "mbed.h"
#include "rtos.h"
#include "st7565LCD.h"
#include "PinDetect.h"
#include "RotaryEncoder.h"
#include "AverageAnalogIn.h"
#include "Sequence.h"
#include "ST7565_SequencerDisplay.h"

#define TITLE_STR1  ("BaseMachine Sequencer")
#define TITLE_STR2  ("20160817")

const int SEQUENCE_N = 16;
const int PITCH_MAX = 12;
const int BPM_MIN = 60;
const int BPM_MAX = 240;

// Devices
//
//ST7565(PinName mosi, PinName sclk, PinName cs, PinName rst, PinName a0);
ST7565 gLCD(PB_15, PB_13, PB_12, PB_2, PB_1);

AverageAnalogIn AinPulseWidth(PC_2);
AverageAnalogIn AinCutOff(PB_0);
AverageAnalogIn AinResonance(PC_1);
AverageAnalogIn AinEnvMod(PC_3);
AverageAnalogIn AinLevel(PC_0);
AverageAnalogIn AinDuration(PA_4);
AverageAnalogIn AinDecay(PA_1);
AverageAnalogIn AinSustain(PA_0);

RotaryEncoder RotEncStep(D2, D3, 0, SEQUENCE_N - 1, 0);
RotaryEncoder RotEncPitch(D4, D5, 0, PITCH_MAX, 0);
RotaryEncoder RotEncBpm(D14, D15, BPM_MIN, BPM_MAX, 120);

PinDetect PinWaveShape(PD_2, PullUp);
PinDetect PinModNumber(PC_11, PullUp);
PinDetect PinOctaveUp(PC_10, PullUp);
PinDetect PinOctaveDown(PC_12, PullUp);
PinDetect PinNoteOnOff(PA_13, PullUp);
PinDetect PinTie(PA_14, PullUp);
PinDetect PinAccent(PA_15, PullUp);
PinDetect PinRunStop(PB_7, PullUp);

// Grobal Variables
//
Sequence sequences[SEQUENCE_N];
ST7565_SequencerDisplay sequencerDisplay(&gLCD, sequences, SEQUENCE_N);

volatile int currentStep = 0;
volatile bool isRunning = false;
volatile bool isDirty = false;
volatile int bpm = 120;
volatile uint8_t pinFlag = 0x00;

enum {
    bWaveShape  = 0x01,
    bModNumber  = 0x02,
    bOctaveUp   = 0x04,
    bOctaveDown = 0x08,
    bNoteOnOff  = 0x10,
    bTie        = 0x20,
    bAccent     = 0x40,
    bRunStop    = 0x80
};

// とりあえずの変数(後でClassのメンバ変数に格納)
#define WAVESHAPE_N 2
#define MOD_NUMBER_MAX 1
volatile int waveShape = 0;
volatile int modNumber = 0;
volatile uint8_t pulseWidth;
volatile uint8_t cutOff;
volatile uint8_t resonance;
volatile uint8_t envMod;
volatile uint8_t level;
volatile uint8_t duration;
volatile uint8_t decay;
volatile uint8_t sustain;

//------------------------------------------------------------------------
// PinDetect ISR
//------------------------------------------------------------------------
void swWaveShapePressed()
{
    pinFlag |= bWaveShape;
}

void swModNumberPressed()
{
    pinFlag |= bModNumber;
}

void swOctaveUpPressed()
{
    pinFlag |= bOctaveUp;
}

void swOctaveDownPressed()
{
    pinFlag |= bOctaveDown;
}

void swNoteOnOffPressed()
{
    pinFlag |= bNoteOnOff;
}

void swTiePressed()
{
    pinFlag |= bTie;
}

void swAccentPressed()
{
    pinFlag |= bAccent;
}

void swRunStopPressed()
{
    pinFlag |= bRunStop;
}

//------------------------------------------------------------------------
// Functions
//------------------------------------------------------------------------
void pollingPots()
{
    //pulseWidth = AinPulseWidth.read_u16() >> 8;
    cutOff     = AinCutOff.read_u16()     >> 8;
    resonance  = AinResonance.read_u16()  >> 8;
    //envMod     = AinEnvMod.read_u16()     >> 8;
    //level      = AinLevel.read_u16()      >> 8;
    duration   = AinDuration.read_u16()   >> 8;
    decay      = AinDecay.read_u16()      >> 8;
    sustain    = AinSustain.read_u16()    >> 8;
}

void pollingRotEncs()
{
    int _bpm = RotEncBpm.getVal();
    if (_bpm != bpm) {
        bpm = _bpm;
        isDirty = true;
    }
    
    int _step = RotEncStep.getVal();
    if (_step != currentStep) {
        currentStep = _step;
        // syncronize sequence value & Rotary Encoder's value
        RotEncPitch.setVal(sequences[currentStep].getPitch());
        isDirty = true;
    }
    
    int _pitch = RotEncPitch.getVal();
    if (_pitch != sequences[currentStep].getPitch()) {
        sequences[currentStep].setPitch(_pitch);
        isDirty = true;
    }
}

void pollingPins()
{
    if (pinFlag & bWaveShape) {
        #if (UART_TRACE)
        printf("PinWaveShape Pushed\r\n");
        #endif
        waveShape++;
        if (waveShape == WAVESHAPE_N) {
            waveShape = 0;
        }
        pinFlag &= ~bWaveShape;
    }
    
    if (pinFlag & bModNumber) {
        #if (UART_TRACE)
        printf("PinModNumber Pushed\r\n");
        #endif
        modNumber++;
        if (modNumber > MOD_NUMBER_MAX) {
            modNumber = 0;
        }
        pinFlag &= ~bModNumber;
    }
    
    if (pinFlag & bOctaveUp) {
        #if (UART_TRACE)
        printf("PinOctaveUp Pushed\r\n");
        #endif
        sequences[currentStep].setOctave(sequences[currentStep].getOctave() + 1);
        pinFlag &= ~bOctaveUp;
    }

    if (pinFlag & bOctaveDown) {
        #if (UART_TRACE)
        printf("PinOctaveDown Pushed\r\n");
        #endif
        sequences[currentStep].setOctave(sequences[currentStep].getOctave() - 1);
        pinFlag &= ~bOctaveDown;
    }
    
    if (pinFlag & bNoteOnOff) {
        #if (UART_TRACE)
        printf("PinNoteOnOff Pushed\r\n");
        #endif
        sequences[currentStep].noteOn = !sequences[currentStep].noteOn;
        pinFlag &= ~bNoteOnOff;
    }
    
    if (pinFlag & bTie) {
        #if (UART_TRACE)
        printf("PinTie Pushed\r\n");
        #endif
        sequences[currentStep].tie = !sequences[currentStep].tie;
        pinFlag &= ~bTie;
    }
    
    if (pinFlag & bAccent) {
        #if (UART_TRACE)
        printf("PinAccent Pushed\r\n");
        #endif
        sequences[currentStep].accent = !sequences[currentStep].accent;
        pinFlag &= ~bAccent;
    }
    
    if (pinFlag & bRunStop) {
        #if (UART_TRACE)
        printf("PinRunStop Pushed\r\n");
        #endif
        isRunning = !isRunning;
        pinFlag &= ~bRunStop;
    }
}

void dumpToLCD()
{
    char buff[64];
    int col = 0;
    
    gLCD.clear();

    sprintf(buff, "Run: %d", isRunning);
    gLCD.drawstring(0, col++, buff);
    sprintf(buff, "BPM: %d", bpm);
    gLCD.drawstring(0, col++, buff);

    sprintf(buff, "Step: %d", currentStep);
    gLCD.drawstring(0, col++, buff);
    sprintf(buff, "NoteOn: %d", sequences[currentStep].noteOn);
    gLCD.drawstring(0, col++, buff);
    sprintf(buff, "Pitch: %d", sequences[currentStep].getPitch());
    gLCD.drawstring(0, col++, buff);
    sprintf(buff, "Octave: %d", sequences[currentStep].getOctave());
    gLCD.drawstring(0, col++, buff);
    sprintf(buff, "Tie: %d", sequences[currentStep].tie);
    gLCD.drawstring(0, col++, buff);
    sprintf(buff, "Accent: %d", sequences[currentStep].accent);
    gLCD.drawstring(0, col++, buff);
    
    col = 0;
    sprintf(buff, "WavS: %d", waveShape);
    gLCD.drawstring(64, col++, buff);
    sprintf(buff, "ModN: %d", modNumber);
    gLCD.drawstring(64, col++, buff);
    
    col = 2;
    sprintf(buff, "PW%3d CO%3d", pulseWidth, cutOff);
    gLCD.drawstring(60, col++, buff);
    sprintf(buff, "RS%3d EV%3d", resonance, envMod);
    gLCD.drawstring(60, col++, buff);
    sprintf(buff, "LV%3d DR%3d", level, duration);
    gLCD.drawstring(60, col++, buff);
    sprintf(buff, "DC%3d ST%3d", decay, sustain);
    gLCD.drawstring(60, col++, buff);

    gLCD.display();
}

//------------------------------------------------------------------------
// Main routine
//------------------------------------------------------------------------
int main()
{
    #if (UART_TRACE)
    printf("*** BaseMachine Sequencer ***\r\n");
    #endif

    gLCD.begin(0x12);
    gLCD.clear();
    gLCD.drawstring(0, 0, TITLE_STR1);
    gLCD.drawstring(0, 1, TITLE_STR2);
    gLCD.display();
    Thread::wait(1000);
    
    RotEncStep.setInterval(100);
    RotEncPitch.setInterval(100);
    RotEncBpm.setInterval(100);
    
    PinWaveShape.attach_asserted(&swWaveShapePressed);
    PinWaveShape.setAssertValue(0);
    PinWaveShape.setSampleFrequency();  
    
    PinModNumber.attach_asserted(&swModNumberPressed);
    PinModNumber.setAssertValue(0);
    PinModNumber.setSampleFrequency();
    
    PinOctaveUp.attach_asserted(&swOctaveUpPressed);
    PinOctaveUp.setAssertValue(0);
    PinOctaveUp.setSampleFrequency();  

    PinOctaveDown.attach_asserted(&swOctaveDownPressed);
    PinOctaveDown.setAssertValue(0);
    PinOctaveDown.setSampleFrequency();
    
    PinNoteOnOff.attach_asserted(&swNoteOnOffPressed);
    PinNoteOnOff.setAssertValue(0);
    PinNoteOnOff.setSampleFrequency();  
    
    PinTie.attach_asserted(&swTiePressed);
    PinTie.setAssertValue(0);
    PinTie.setSampleFrequency();  
    
    PinAccent.attach_asserted(&swAccentPressed);
    PinAccent.setAssertValue(0);
    PinAccent.setSampleFrequency();    
    
    PinRunStop.attach_asserted(&swRunStopPressed);
    PinRunStop.setAssertValue(0);
    PinRunStop.setSampleFrequency(); 
    
    #if (UART_TRACE)
    printf("Setup Devices OK\r\n");
    #endif
    
    for (;;) {
        pollingPots();
        pollingPins();
        pollingRotEncs();
        dumpToLCD();
    }  
}
