/*
 * main.cpp
 * BaseMachine Sequencer
 *
 * mbed Rev 135 / mbed-rtos Rev 123
 *
 * 2017.02.14 mbed Rev 135 / mbed-rtos Rev 123に変更
 * 2016.11.07 UIを統合
 *
 */

#include "mbed.h"
#include "rtos.h"

#define UART_TRACE  (0)
#include "BaseMachineCommon.h"
#include "BaseMachineUIController.h"
#include "SequenceSender.h"
#include "EnvelopeGenerator.h"
#include "SpiOscController.h"
#include "SpiAmpController.h"
#include "SpiFilterController.h"
#include "SpiVcvsLpfController.h"
#include "SpiSvfController.h"

#define SPI1_RATE   (1000000)
#define I2C1_RATE   (100000)

const int samplingPeriod = 1;   // ms
const int bpm = 120;
const int envelopeLength = (60 * 1000 / (bpm * 4)) / samplingPeriod;

// Devices
//
BaseMachineUIController UIController;

//SPI1 (PinName mosi, PinName miso, PinName sclk, PinName ssel=NC)
SPI SpiMaster(PA_7, PA_6, PA_5);
DigitalOut SpiMasterReset(PB_10, 0);

Sequence sequences[SEQUENCE_N];
SequenceSender SequenceSender(sequences, SEQUENCE_N, samplingPeriod, bpm);

Envelope envelope(4095, envelopeLength, envelopeLength*3/4, envelopeLength/2, 2047);
EnvelopeGenerator EnvelopeGenerator;

SpiOscController OscController(&SpiMaster, D9);
SpiAmpController AmpController(&SpiMaster, D8, D7);

#if USE_VCVS
    SpiVcvsLpfController FilterController(&SpiMaster, D10);
#elif USE_SVF
    SpiSvfController FilterController(&SpiMaster, D10);
#endif

InterruptIn SyncIn(PC_4);
//DigitalOut led(D13);

// Grobal Variables
//
int playingStep = 0;
int editingStep = 0;
bool isRunning = false;

//------------------------------------------------------------------------
// Callback functions
//------------------------------------------------------------------------
void updateTicks(int ticks)
{
    uint16_t level = EnvelopeGenerator.update(sequences[SequenceSender.getStep()].isTie());
    
    // Amp
    uint16_t ampLevel = level;
    if (sequences[SequenceSender.getStep()].isNoteOn())
    {
        ampLevel = ampLevel >> 1;
        if (sequences[SequenceSender.getStep()].isAccent())
        {
            ampLevel = ampLevel * (1.0f + ((float)UIController.getAccentLevel()) / 128.0f);
        }
    }
    else
    {
        ampLevel = 0;
    }
    AmpController.outDca(ampLevel);
    
    // Oscillator
    OscController.setFrequency10(SequenceSender.getFrequency10());
    OscController.outDco();
    
    // Filter Env Mod
    uint16_t cutoff16 = FilterController.getCutoff();
    uint8_t resonance = FilterController.getResonance();
    uint8_t mod = FilterController.getModLevel();
    float modCoeff = (float)mod / (float)(cutoff16 << 1);
    cutoff16 = cutoff16 * (1.0f + ((float)level / 0x8000) * modCoeff);
    if (cutoff16 > 255) {
        cutoff16 = 255;
    }
    FilterController.outDcf(cutoff16, resonance);
    
    if (ticks == 0)
    {
        EnvelopeGenerator.init(envelope);
        playingStep = SequenceSender.getStep();
        UIController.setPlayingStep(playingStep);
    }
}

//------------------------------------------------------------------------
// Functions
//------------------------------------------------------------------------
void setParams()
{
    UIController.getSequences(&sequences);

    SequenceSender.setBpm(UIController.getBpm());

    OscillatorParam osc;
    UIController.getOscillatorParam(&osc);
    OscController.setWaveShape(osc.waveShape);
    OscController.setPulseWidth(osc.pulseWidth << 1);

    FilterParam flt;
    UIController.getFilterParam(&flt);
    FilterController.setCutoff(flt.cutoff << 1);
    FilterController.setResonance(flt.resonance << 1);

    EnvelopeParam env;
    UIController.getEnvelopeParam(&env);
    //envelope.setLevel(env.level << 5);
    FilterController.setModLevel(env.level << 1);
    //ToDo: Impl. envelope length
    envelope.setLength(envelopeLength);
    envelope.setDuration((float)env.duration / 128.0f * envelopeLength);
    envelope.setDecay((float)env.decay / 128.0f * envelopeLength);
    envelope.setSustain(env.sustain << 5);

    bool _isRunning = UIController.getIsRunning();
    if (_isRunning != isRunning)
    {
        isRunning = _isRunning;
        if (isRunning)
        {
            playingStep = 0;
            SequenceSender.run(playingStep);
        }
        else
        {
            AmpController.outDca(0);
            SequenceSender.stop();
        }
    }
}

void syncFunction()
{
    //led = !led;
    SequenceSender.sync();
}

//------------------------------------------------------------------------
// Main routine
//------------------------------------------------------------------------
int main()
{
    #if (UART_TRACE)
    printf("*** BaseMachine Sequencer ***\r\n");
    #endif

    //--------------------------------------------------------------------
    // Initialize objects
    //
    Sequence::setBaseNoteNumber(baseNoteNumber);
    
    //--------------------------------------------------------------------
    // Setup Devices
    //
    SpiMasterReset = 1;
    SpiMaster.format(8, 0);
    SpiMaster.frequency(SPI1_RATE);
    
    // Mute output
    AmpController.outDca(0);
    
    UIController.init();
    setParams();
    
    EnvelopeGenerator.init(envelope);
    
    // Sync Interrupt
    SyncIn.rise(&syncFunction);

    SequenceSender.attachUpdate(&updateTicks);

    //--------------------------------------------------------------------
    // Main loop
    //
    for (;;)
    {
        UIController.update();
        setParams();
        
        Thread::wait(10);
    }
}
