// Library for the DS1620 digital thermometer
// Copyright (C) <2015> Ryan Bancroft
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// For copy of the GNU General Public License
// see <http://www.gnu.org/licenses/>.


#ifndef DS1620_H
#define DS1620_H

#include "mbed.h"

enum clock_frequencies_t {freq500k = 1, 
                          freq250k = 2,
                          freq125k = 4,
                          freq20k = 25,
                          freq10k = 50,
                          freq1k = 500}; 
                          
#define CLOCK_DELAY freq500k

// Command definitions
#define READ_TEMPERATURE 0xAA
#define WRITE_TH         0x01
#define WRITE_TL         0x02
#define READ_TH          0xA1
#define READ_TL          0xA2
#define READ_COUNTER     0xA0
#define READ_SLOPE       0xA9
#define START_CONVERT    0xEE
#define STOP_CONVERT     0x22
#define WRITE_CONFIG     0x0C
#define READ_CONFIG      0xAC
#define LOAD_SLOPE       0x41

/**
* Example:
* @code
* // Send the temperature to your computer
* #include "mbed.h"
* #include "DS1620.h"
*
* DS1620 temperatureSensor(p11, p12, p13);
* Serial pc(USBTX, USBRX);
*
* int main() {
*   //Set mode to CPU & One Shot if it's not already set.
*   if((temperatureSensor.readConfig() & 0x03) != 0x03) {
*      temperatureSensor.writeConfig(0x03);
*   }
*   
*   float temperature = 0.0;
*   while(1) {
*       temperatureSensor.startConversion();
*       wait(1.0); //Wait for the temperature conversion to complete.  You can also poll the DONE flag in the STATUS REGISTER
*       temperature = temperatureSensor.getHighResolutionTemperature();
*       pc.printf("%.2f C\r\n", temperature);
*   }       
* }
* @endcode
*/

// Class declaration
class DS1620
{
    public:
        DS1620 (PinName dq, PinName clk, PinName rst);
        ~DS1620 ();
 
 
        /**
        * Get the temperature in degrees Celsius
        * 
        * @returns The temperature in Celsius with 0.5 degree resolution as a float 
        */
        float getTemperature();
        
        
        /**
        * Get the temperature in degrees Celsius
        * 
        * @returns The temperature in Celsius calculated using the COUNTER and SLOPE REGISTERS as a float
        * @note The DS1620 must be in 1SHOT mode for accurate results.
        */
        float getHighResolutionTemperature();
 
        /**
        * Read the TEMPERATURE REGISTER
        * 
        * @returns The temperature in Celsius with 0.5 degree resolution as stored in the DS1620 registers 
        */       
        unsigned short readTemperatureRaw();
        
        /**
        * Read the CONFIGURATION/STATUS REGISTER
        * 
        * @returns The value of the CONFIGURATION/STATUS REGISTER 
        */
        unsigned char readConfig();
        
        /**
        * Write to the CONFIGURATION/STATUS REGISTER
        * 
        * @param  config  The value to write to the CONFIGURATION/STATUS REGISTER 
        */
        void writeConfig(unsigned char config);
        

        /**
        * Read the TL (Thermostat Low Limit) REGISTER
        * 
        * @returns  The value to write to the TL (Thermostat Low Limit) REGISTER in the DS1620 format
        */        
        unsigned short readTLRaw();
        
        /**
        * Write to the TL (Thermostat Low Limit) REGISTER
        * 
        * @param  temperature  The value to write to the TL (Thermostat Low Limit) REGISTER in the DS1620 format
        */
        void writeTLRaw(unsigned short temperature);
        
        
        /**
        * Read the TH (Thermostat High Limit) REGISTER
        * 
        * @returns  The value to write to the TH (Thermostat High Limit) REGISTER in the DS1620 format
        */
        unsigned short readTHRaw();
        
        
        /**
        * Write to the TH (Thermostat High Limit) REGISTER
        * 
        * @param  temperature  The value to write to the TH (Thermostat High Limit) REGISTER in the DS1620 format
        */
        void writeTHRaw(unsigned short temperature);
        
        /**
        * Read the COUNTER REGISTER
        * 
        * @returns The COUNTER REGISTER
        */
        unsigned short readCounter();
        
        /**
        * Read the SLOPE REGISTER
        * 
        * @returns The SLOPE REGISTER
        */
        unsigned short readSlope();
        
        
        /**
        * Loads the SLOPE REGISTER into the COUNTER REGISTER
        */
        void loadSlope();
        
        
        /**
        * Starts a temperature conversion
        */        
        void startConversion();
        
        
        /**
        * Stops a temperature conversion
        */
        void stopConversion();
        
        /**
        * Sets the clock frequency for the serial interface
        *
        * @param frequency (freq500k, freq250k, freq125k, freq20k, freq10k, freq1k)
        * @note Defaults to 500KHz (freq500k)
        */
        void setSerialClockFrequency(clock_frequencies_t frequency);
        

    private:
        void shiftOut(unsigned char data);
        unsigned char shiftIn();
        
        clock_frequencies_t _clockDelay;
        
        DigitalInOut _dq;
        DigitalOut _clk;
        DigitalOut _rst;
        
};

#endif