// Library for the DS1620 digital thermometer
// Copyright (C) <2015> Ryan Bancroft
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// For copy of the GNU General Public License
// see <http://www.gnu.org/licenses/>.

#include "mbed.h"
#include "DS1620.h"

#define WRITE_COMPLETE_TIME_MS   (100)        // time for write to complete (100 msec) 

// Constructor
DS1620::DS1620(PinName dq, PinName clk, PinName rst) : _dq(dq), _clk(clk), _rst(rst)
{
    // Initialize members  
    _dq.input(); 
    _clk = 1;
    _rst = 0;
    _clockDelay = CLOCK_DELAY;
}

// Destructor
DS1620::~DS1620()
{

}

float DS1620::getTemperature()
{
    float temp = 0.0;
    unsigned short rawTemp = readTemperatureRaw();
    temp = rawTemp / 2.0f;
    if (rawTemp >= 0x100) {
        temp -= 256.0f;
    }
    return temp;
}

float DS1620::getHighResolutionTemperature()
{
    float temp = 0.0f;
    unsigned short rawTemp, counter, slope = 0;
    rawTemp = readTemperatureRaw();
    counter = readCounter();
    loadSlope();
    slope = readCounter();
    
    temp = (rawTemp >> 1) - 0.25f;
    if (rawTemp >= 0x100) {
        temp -= 256.0f;
    }
    temp += ((float)slope - (float)counter) / (float)slope;   
    return temp;
}

unsigned short DS1620::readTemperatureRaw()
{
    unsigned short temp = 0;
    unsigned char tempLSB, tempMSB = 0;
    
    _rst = 1;
    shiftOut(READ_TEMPERATURE);
    tempLSB = shiftIn();
    tempMSB = shiftIn();
    _rst = 0;
    temp = tempMSB * 256;
    temp += tempLSB;
    return temp;
}
        
unsigned char DS1620::readConfig()
{
    unsigned char data = 0;
    _rst = 1;
    shiftOut(READ_CONFIG);
    data = shiftIn();
    _rst = 0;
    return data;
}

void DS1620::writeConfig(unsigned char config)
{
    _rst = 1;
    shiftOut(WRITE_CONFIG);
    shiftOut(config);
    _rst = 0;
    wait_ms(WRITE_COMPLETE_TIME_MS);  // wait for write to complete
}
    
unsigned short DS1620::readTLRaw()
{
    unsigned short temp = 0;
    unsigned char tempLSB, tempMSB = 0;
    
    _rst = 1;
    shiftOut(READ_TL);
    tempLSB = shiftIn();
    tempMSB = shiftIn();
    _rst = 0;
    temp = tempMSB * 256;
    temp += tempLSB;
    return temp;
}

void DS1620::writeTLRaw(unsigned short temperature)
{
    unsigned char tempMSB, tempLSB = 0;
    tempLSB = temperature & 0xFF;
    tempMSB = (temperature >> 8) & 0xFF;
    _rst = 1;
    shiftOut(WRITE_TL);
    shiftOut(tempLSB);
    shiftOut(tempMSB);
    _rst = 0;
    wait_ms(WRITE_COMPLETE_TIME_MS);  // wait for write to complete
}   
    
        
unsigned short DS1620::readTHRaw()
{
    unsigned short temp = 0;
    unsigned char tempLSB, tempMSB = 0;
    
    _rst = 1;
    shiftOut(READ_TH);
    tempLSB = shiftIn();
    tempMSB = shiftIn();
    _rst = 0;
    temp = tempMSB * 256;
    temp += tempLSB;
    return temp;
}

void DS1620::writeTHRaw(unsigned short temperature)
{
    unsigned char tempMSB, tempLSB = 0;
    tempLSB = temperature & 0xFF;
    tempMSB = (temperature >> 8) & 0xFF;
    _rst = 1;
    shiftOut(WRITE_TH);
    shiftOut(tempLSB);
    shiftOut(tempMSB);
    _rst = 0;
    wait_ms(WRITE_COMPLETE_TIME_MS);  // wait for write to complete
}
        
unsigned short DS1620::readCounter()
{
    unsigned short temp = 0;
    unsigned char tempLSB, tempMSB = 0;
    
    _rst = 1;
    shiftOut(READ_COUNTER);
    tempLSB = shiftIn();
    tempMSB = shiftIn();
    _rst = 0;
    temp = tempMSB * 256;
    temp += tempLSB;
    return temp;
}
        
unsigned short DS1620::readSlope()
{
    unsigned short temp = 0;
    unsigned char tempLSB, tempMSB = 0;
    
    _rst = 1;
    shiftOut(READ_SLOPE);
    tempLSB = shiftIn();
    tempMSB = shiftIn();
    _rst = 0;
    temp = tempMSB * 256;
    temp += tempLSB;
    return temp;
}

void DS1620::loadSlope()
{
    _rst = 1;
    shiftOut(LOAD_SLOPE);
    _rst = 0;
}
        
void DS1620::startConversion()
{
    _rst = 1;
    shiftOut(START_CONVERT);
    _rst = 0;
}

void DS1620::stopConversion()
{
    _rst = 1;
    shiftOut(STOP_CONVERT);
    _rst = 0;
}

void DS1620::setSerialClockFrequency(clock_frequencies_t frequency) 
{
    _clockDelay = frequency;
}

unsigned char DS1620::shiftIn()
{
    unsigned char temp = 0;
    unsigned char data = 0;
    _dq.input();
    for(int x=0;x<8;x++) {
        _clk = 0;
        wait_us(_clockDelay);
        temp = _dq;
        temp = temp << 7;
        data = data >> 1;
        data = data | temp;
        _clk = 1;
        wait_us(_clockDelay);
    }
    return data;   
}

void DS1620::shiftOut(unsigned char data)
{
    _dq.output();
    for(int x=0;x<8;x++){
        _clk = 0;
        _dq = data & 0x01;
        wait_us(_clockDelay);
        _clk = 1;
        wait_us(_clockDelay);
        data = data >> 1;
    }
    _dq.input();
}