#include "mbed.h"
#include "string"
#include "list"
#include "WS2811.h"
#include "Colors.h"
#include "TSISensor.h"
#include "MMA8451Q.h"
#include "mpr121.h"
#include "statechart.h"

#define MMA8451_I2C_ADDRESS (0x1d<<1)

/*-----------------------------------------------------------------------*/

// Neopixel Setup
unsigned const nLEDs = MAX_LEDS_PER_STRIP;          // Defined in WS2811.h
unsigned const DATA_OUT_PIN1 = 2;                   // PTD2
unsigned const DATA_OUT_PIN2 = 3;                   // PTD3

// Maestro Setup
Serial maestro(USBTX,USBRX);

// Timer Setup
Timer timeRunning;

// MPR121 Setup
InterruptIn interrupt(D9);
I2C i2c(D14, D15);
Mpr121 mpr121(&i2c, Mpr121::ADD_VSS); // constructor(i2c object, i2c address of the mpr121)
int touch_val = 0;
int prox_val = 0;

// Accelerometer Setup
MMA8451Q acc(PTE25, PTE24, MMA8451_I2C_ADDRESS);

// Battery Read Setup
AnalogIn battery(A1);

/*-----------------------------------------------------------------------*/

void setToProxMode( bool prox ) {
    
    if( prox ) {                                                            // Proximity Mode Setup
    
        mpr121.setProximityMode(true);
        unsigned char mode = mpr121.read(ELE_CFG);                          // Get the current mode
        mpr121.write(ELE_CFG,0x00);                                         // Put the MPR into setup mode
        mpr121.write(EPROXTTH, 0x00);                                       // Write the new threshold
        mpr121.write(EPROXRTH, 0x00);
        mpr121.write(ELE_CFG, mode);                                        //Restore the operating mode
        
    } else {                                                                // Touch Mode Setup
    
        mpr121.setProximityMode(false);
        unsigned char badTouch[12]     = {  0x05, 0x05, 0x05,               // Tail-side Body
                                            0x05, 0x05, 0x05,               // Tail-side End
                                            0x07, 0x07, 0x07,               // Head-side Body
                                            0x05, 0x05, 0x05,};             // Head-side End
    
        unsigned char sweetRelease[12] = {  0x02, 0x02, 0x02,               // Tail-side Body
                                            0x02, 0x02, 0x02,               // Tail-side End
                                            0x02, 0x02, 0x02,               // Head-side Body
                                            0x02, 0x02, 0x02};              // Head-side End
        for( int i = 0; i < 12; i++ ) {                             
            mpr121.setElectrodeThreshold(i,badTouch[i],sweetRelease[i]);    // Reset Touch Sensitivity
        }
    }
}

// Primary Interrupt Function for MPR121
void fallInterrupt() {
    touch_val = mpr121.read(0x00);
    touch_val += mpr121.read(0x01) << 8;
}

/*-----------------------------------------------------------------------*/

int main() {
    
    WS2811 lightStrip1(nLEDs, DATA_OUT_PIN1);
    WS2811 lightStrip2(nLEDs, DATA_OUT_PIN2);
    lightStrip1.begin();
    lightStrip2.begin();

    Statechart statechart;

    // Set to Touch Mode
    setToProxMode(false);
    
    // Start MPR121 Interrupts
    interrupt.fall(&fallInterrupt);
    interrupt.mode(PullUp);
    
    // Main Loop: Show Neopixels
    timeRunning.start();
    float currTime;
    float xyz[3];
    
    while(true) {
        currTime = timeRunning.read_ms();
        acc.getAccAllAxis(xyz);
        
        statechart.fishStatechart(currTime, touch_val, prox_val, xyz, maestro, lightStrip1, lightStrip2, battery);
    }
}

