/* mbed Joystick Library
 * Copyright (c) 2015, rwunderl, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "Joystick.h"

// apply the deadzone to the axis extremes (comment if you don't want to use this feature)
#define JOYSTICK_DEADZONE_EXTREMES

// adjust the deadzone value for your application, if required
const double JOYSTICK_DEADZONE = 0.01;
// set up these constants so they don't have to be calculated every time
#ifdef JOYSTICK_DEADZONE_EXTREMES
    const double JOYSTICK_MIN    = 0.0 + JOYSTICK_DEADZONE;    // effective minimum joystick value
    const double JOYSTICK_MINMID = 0.5 - JOYSTICK_DEADZONE;    // minimum-side midpoint value
    const double JOYSTICK_MAXMID = 0.5 + JOYSTICK_DEADZONE;    // maximum-side midpoint value
    const double JOYSTICK_MAX    = 1.0 - JOYSTICK_DEADZONE;    // effective maximum joystick value
#else
    const double JOYSTICK_MIN    = 0.0;    // minimum joystick value
    const double JOYSTICK_MINMID = 0.5 - JOYSTICK_DEADZONE;    // minimum-side midpoint value
    const double JOYSTICK_MAXMID = 0.5 + JOYSTICK_DEADZONE;    // maximum-side midpoint value
    const double JOYSTICK_MAX    = 1.0;    // maximum joystick value
#endif

double _map(double value, double in_min, double in_max, double out_min, double out_max) {
    return (value - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

double _scale(double pos) {
    double scaled = 0;

    if (pos < JOYSTICK_MINMID) {
        scaled = _map(pos, JOYSTICK_MIN, JOYSTICK_MINMID, -1, 0);
    } else if (pos > JOYSTICK_MAXMID) {
        scaled = _map(pos, JOYSTICK_MAXMID, JOYSTICK_MAX, 0, 1);
    }

    // check out of bounds [-1.0, 1.0] (happens at the extremes)
    if (scaled < -1) {
        scaled = -1;
    } else if (scaled > 1) {
        scaled = 1;
    }

    return scaled;
}

Joystick::Joystick(PinName x, PinName y, PinName a) : _x(x), _y(y), _a(a) {
}

joypos Joystick::read() {
    joypos pos;

    pos.x = getX();
    pos.y = getY();

    return pos;
}

double Joystick::getX() {
    double val = _x.read();
    val = _scale(val);
    return val;
}

double Joystick::getY() {
    double val = _y.read();
    val = _scale(val);
    return val;
}

double Joystick::getRawX() {
    return _x.read();
}

double Joystick::getRawY() {
    return _y.read();
}

void Joystick::rise(void (*fptr)(void)) { 
    _a.rise(fptr);
}

void Joystick::fall(void (*fptr)(void)) {
    _a.fall(fptr);
}
