/* mbed Joystick Library
 * Copyright (c) 2015, rwunderl, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_JOYSTICK_H
#define MBED_JOYSTICK_H

#include "mbed.h"

/** Joystick position structure
 *
 * Struct to keep track of the joystick x and y positions.
 */
struct joypos {
    double x;
    double y;
};

/** Joystick class
 *
 * Reads a 2-axis analog joystick and keeps track of its positions.
 *
 * The joystick uses 2 AnalogIns and 1 DigitalIn
 *     - X AnalogIn
 *     - Y AnalogIn
 *     - A DigitalIn (momentary pushbutton; Low is pressed)
 */
class Joystick {
public:
    /** Joystick interface
     * Create the joystick object and assign pins.
     *
     * @param x X-axis AnalogIn.
     * @param y Y-axis AnalogIn.
     * @param a Pushbutton DigitalIn.
     */ 
    Joystick(PinName x, PinName y, PinName a);

    /** Read the joystick position
     * The position is returned as a joypos value with x and y in the range [-1.0, 1.0].
     *
     * @returns Position structure of the joystick.
     */
    joypos read();

    /** Get the X position
     * Read the joystick's horizontal position, represented as a double value in the range [-1.0, 1.0].
     *
     * @returns Horizontal position of the joystick.
     */
    double getX();

    /** Get the Y position
     * Read the joystick's vertical position, represented as a double value in the range [-1.0, 1.0].
     *
     * @returns Vertical position of the joystick.
     */
    double getY();

    /** Get the raw X position
     * Read the joystick's raw horizontal position, represented as a double value in the range [0.0, 1.0].
     *
     * @returns Horizontal position of the joystick.
     */
    double getRawX();

    /** Get the raw Y position
     * Read the joystick's raw vertical position, represented as a double value in the range [0.0, 1.0].
     *
     * @returns Vertical position of the joystick.
     */
    double getRawY();

    /** Attach the rise interrupt
     * Attach a function pointer to call when a rising edge occurs on the button input.
     *
     * @param fptr Pointer to a void function. Set to NULL or 0 for none.
     */
    void rise(void (*fptr)(void));

    /** Attach the fall interrupt
     * Attach a function pointer to call when a falling edge occurs on the button input.
     *
     * @param fptr Pointer to a void function. Set to NULL or 0 for none.
     */
    void fall(void (*fptr)(void));

protected:
    AnalogIn _x;
    AnalogIn _y;
    InterruptIn _a;
};

#endif /* MBED_JOYSTICK_H */
