



#ifndef _ADAFRUIT_PWMServoDriver_H
#define _ADAFRUIT_PWMServoDriver_H

#include "mbed.h"
#include <cmath>

#define PCA9685_SUBADR1 0x2
#define PCA9685_SUBADR2 0x3
#define PCA9685_SUBADR3 0x4

#define PCA9685_MODE1 0x0
#define PCA9685_MODE2 0x1
#define PCA9685_PRESCALE 0xFE

#define LED0_ON_L 0x6
#define LED0_ON_H 0x7
#define LED0_OFF_L 0x8
#define LED0_OFF_H 0x9

#define ALLLED_ON_L 0xFA
#define ALLLED_ON_H 0xFB
#define ALLLED_OFF_L 0xFC
#define ALLLED_OFF_H 0xFD



class PCA9685Lib; //Forward declaration

/*! \struct LEDarr
    \brief LEDarr is a simple struct that stores the outpin it is, 
    *a pointer to the class it is in, and an overloaded = operator 
    * to make assigning duty cycle easier.
    A more detailed class description.
*/
struct LEDarr {

    uint8_t LEDnum;
    PCA9685Lib* PCALib;
    /** Overloaded assignement operator.
    * Allows the user to more simply assign duty cycle values to a pin. 
    *   @param duty unsigned 16 integer as the duty cycle 
    */
    void operator= (uint16_t duty);
};

/** My PCA9685Lib class
*  used for controlling the pca9685
*
* Example:
* @code
* // Change the duty cycle of an led
*
* #include "mbed.h"
* #include "PCA9685Lib.h"
* I2C i2c(p28, p27);
* PCA9685Lib LedDriver(i2c);
* LedDriver.begin();
* LedDriver.setPWMFreq(1600); 
* int main() 
* {
*     while(1) 
*   {
*
*           for(uint16_t i = 0; i < 16; i++)
            {
                for(uint16_t j = 0; j < 4096; j += 8)
                {
                    LedDriver.LED[i] = j;
                }
            }
            for(uint16_t i = 0; i < 16; i++)
            {
                for(uint16_t j = 4095; j > 0; j -= 8)
                {
                    LedDriver.LED[i] = j;
                }
            }
*        
*     }
* }
* @endcode
*/


class PCA9685Lib
{
public:
    /** Create an PCA9685Lib instance, pass in an I2C object and the PCA9685 device address
        * The default address is 0x80
        * @param i2cobj Pass and instantiated I2C object to the class to communicate with
        * @param addr pass the device address the constructor (optional)
        *  
        */
    PCA9685Lib(I2C i2cobj, int addr = 0x80); //0b 1_000000_(R/W) <- default slave adress
    /** Seaches for I2C devices
    
           */
    void i2c_probe(void);
    /** Resets the PCA9685 (clears device registers)
           */
    void begin(void);
    /** Sets the I2C frequency.  
           */
    void setI2Cfreq(int freq);
    /** Resets the PCA9685 (clears device registers)
           */
    void reset(void);
    /** sets the pwm frequency
           */
    void setPWMFreq(float freq);
    /** Set the prescale registers on the pca9685
           */
    void setPrescale(uint8_t prescale);
    /** Sets the Mode2 to 0x15 (totem pole configuration) by default
    @param val Mode2 register value
    */
    void setMode2(uint8_t val = 0x15);
    /** Sets the pwm on a specific pin
    * Each PWM clock cycle is divided in 4096 steps
    * See datasheet for more details
    * @param num Output pin number (0-15)
    * @param on This is speicify at what point in the cycle the output will turn on (0-4095 or 0x000-0xFFF)
    * @param off This is speicify at what point in the cycle the output will turn off (0-4095 or 0x000-0xFFF)
           */
    void setPWM(uint8_t num, uint16_t on, uint16_t off);
    /** Sets the duty cycle on a specific pin
    * @param num Output pin number (0-15)
    * @param duty This is speicify at what point in the cycle the output will turn off (0-4095 or 0x000-0xFFF)
           */
    void setDuty(uint8_t num, uint16_t duty);


    LEDarr LED[15];
private:
    int _i2caddr;
    I2C i2c;
    uint8_t read8(char addr);
    void write8(char addr, char d);
};


#endif
/*
  This is a library for our Adafruit 16-channel PWM & Servo driver

  Pick one up today in the adafruit shop!
  ------> http://www.adafruit.com/products/815

  These displays use I2C to communicate, 2 pins are required to
  interface. For Arduino UNOs, thats SCL -> Analog 5, SDA -> Analog 4

  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.
  BSD license, all text above must be included in any redistribution
 ****************************************************/

 /*****************************
  This program was ported from https://github.com/adafruit/Adafruit-PWM-Servo-Driver-Library.

 */
